# Telegram Bot API Documentation

## Overview

This document provides detailed information about the TelegramBot class and its methods for interacting with the Telegram Bot API. The class is designed to simplify the process of sending messages, handling updates, and managing users in a Telegram bot application.

## TelegramBot Class

### Constructor

```php
__construct($token, $db, $debug = false)
```

**Parameters:**
- `$token` (string): The Telegram Bot API token
- `$db` (PDO): Database connection object
- `$debug` (bool, optional): Enable debug mode (default: false)

**Description:**
Initializes a new TelegramBot instance with the provided token and database connection.

### Core Methods

#### processUpdate

```php
processUpdate($update)
```

**Parameters:**
- `$update` (array): The update received from Telegram

**Description:**
Processes an update from Telegram, handling messages and callback queries.

#### saveUser

```php
saveUser($user)
```

**Parameters:**
- `$user` (array): User data from Telegram

**Returns:**
bool: Success status

**Description:**
Saves or updates user information in the database.

### Message Methods

#### sendMessage

```php
sendMessage($chatId, $text, $options = [])
```

**Parameters:**
- `$chatId` (int): Chat ID to send the message to
- `$text` (string): Message text
- `$options` (array, optional): Additional options

**Returns:**
array|bool: Response from API or false on failure

**Description:**
Sends a text message to a chat.

**Options:**
- `parse_mode` (string): Message parsing mode (Markdown, HTML)
- `reply_markup` (array): Keyboard markup
- `reply_to_message_id` (int): Message ID to reply to
- `disable_web_page_preview` (bool): Disable link previews
- `disable_notification` (bool): Send message silently

#### editMessageText

```php
editMessageText($chatId, $messageId, $text, $options = [])
```

**Parameters:**
- `$chatId` (int): Chat ID
- `$messageId` (int): Message ID to edit
- `$text` (string): New message text
- `$options` (array, optional): Additional options

**Returns:**
array|bool: Response from API or false on failure

**Description:**
Edits the text of an existing message.

#### sendPhoto

```php
sendPhoto($chatId, $photo, $caption = '', $options = [])
```

**Parameters:**
- `$chatId` (int): Chat ID
- `$photo` (string): Photo path or file_id
- `$caption` (string, optional): Caption text
- `$options` (array, optional): Additional options

**Returns:**
array|bool: Response from API or false on failure

**Description:**
Sends a photo to a chat.

#### sendDocument

```php
sendDocument($chatId, $document, $caption = '', $options = [])
```

**Parameters:**
- `$chatId` (int): Chat ID
- `$document` (string): Document path or file_id
- `$caption` (string, optional): Caption text
- `$options` (array, optional): Additional options

**Returns:**
array|bool: Response from API or false on failure

**Description:**
Sends a document to a chat.

### Keyboard Methods

#### sendInlineKeyboard

```php
sendInlineKeyboard($chatId, $text, $buttons, $options = [])
```

**Parameters:**
- `$chatId` (int): Chat ID
- `$text` (string): Message text
- `$buttons` (array): Array of inline keyboard buttons
- `$options` (array, optional): Additional options

**Returns:**
array|bool: Response from API or false on failure

**Description:**
Sends a message with an inline keyboard.

#### answerCallbackQuery

```php
answerCallbackQuery($callbackQueryId, $text = '', $showAlert = false)
```

**Parameters:**
- `$callbackQueryId` (string): Callback query ID
- `$text` (string, optional): Text to show to the user
- `$showAlert` (bool, optional): Show as alert instead of notification

**Returns:**
array|bool: Response from API or false on failure

**Description:**
Answers a callback query from an inline keyboard button.

### User Management Methods

#### blockUser

```php
blockUser($userId)
```

**Parameters:**
- `$userId` (int): Telegram user ID

**Returns:**
bool: Success status

**Description:**
Blocks a user from using the bot.

#### unblockUser

```php
unblockUser($userId)
```

**Parameters:**
- `$userId` (int): Telegram user ID

**Returns:**
bool: Success status

**Description:**
Unblocks a previously blocked user.

#### getUserInfo

```php
getUserInfo($userId)
```

**Parameters:**
- `$userId` (int): Telegram user ID

**Returns:**
array|bool: User data or false if not found

**Description:**
Retrieves user information from the database.

### Broadcasting Methods

#### broadcastMessage

```php
broadcastMessage($text, $options = [], $filter = [])
```

**Parameters:**
- `$text` (string): Message text
- `$options` (array, optional): Additional options
- `$filter` (array, optional): Filter criteria for recipients

**Returns:**
array: Array with success count, fail count, and errors

**Description:**
Sends a message to multiple users.

**Filter options:**
- `is_blocked` (bool): Filter by blocked status
- `language_code` (string): Filter by language code
- `last_activity` (string): Filter by last activity date

### Webhook Methods

#### setWebhook

```php
setWebhook($url)
```

**Parameters:**
- `$url` (string): Webhook URL

**Returns:**
array|bool: Response from API or false on failure

**Description:**
Sets the webhook URL for receiving updates.

#### deleteWebhook

```php
deleteWebhook()
```

**Returns:**
array|bool: Response from API or false on failure

**Description:**
Deletes the webhook.

#### getWebhookInfo

```php
getWebhookInfo()
```

**Returns:**
array|bool: Webhook information or false on failure

**Description:**
Gets information about the current webhook.

### Bot Information Methods

#### getMe

```php
getMe()
```

**Returns:**
array|bool: Bot information or false on failure

**Description:**
Gets information about the bot.

#### getBotCommands

```php
getBotCommands()
```

**Returns:**
array|bool: List of bot commands or false on failure

**Description:**
Gets the list of commands registered for the bot.

#### setBotCommands

```php
setBotCommands($commands)
```

**Parameters:**
- `$commands` (array): Array of command objects with 'command' and 'description' keys

**Returns:**
array|bool: Response from API or false on failure

**Description:**
Sets the list of commands for the bot.

### Utility Methods

#### makeRequest

```php
makeRequest($method, $parameters = [])
```

**Parameters:**
- `$method` (string): Telegram API method
- `$parameters` (array, optional): Method parameters

**Returns:**
array|bool: Response from API or false on failure

**Description:**
Makes a request to the Telegram Bot API.

#### formatMessageText

```php
formatMessageText($text)
```

**Parameters:**
- `$text` (string): Text to format

**Returns:**
string: Formatted text

**Description:**
Formats message text for Telegram Markdown.

#### logDebug

```php
logDebug($message)
```

**Parameters:**
- `$message` (string): Debug message

**Description:**
Logs a debug message if debug mode is enabled.

## Examples

### Sending a Simple Message

```php
$bot = new TelegramBot(BOT_TOKEN, $db);
$bot->sendMessage(123456789, 'Hello, world!');
```

### Sending a Message with an Inline Keyboard

```php
$bot = new TelegramBot(BOT_TOKEN, $db);

$buttons = [
    [['text' => 'Button 1', 'callback_data' => 'action:1']],
    [['text' => 'Button 2', 'callback_data' => 'action:2']]
];

$bot->sendInlineKeyboard(123456789, 'Please select an option:', $buttons);
```

### Processing a Callback Query

```php
$bot = new TelegramBot(BOT_TOKEN, $db);

// In your webhook handler
$update = json_decode(file_get_contents('php://input'), true);

if (isset($update['callback_query'])) {
    $callbackQuery = $update['callback_query'];
    $chatId = $callbackQuery['message']['chat']['id'];
    $messageId = $callbackQuery['message']['message_id'];
    $data = $callbackQuery['data'];
    
    // Answer the callback query
    $bot->answerCallbackQuery($callbackQuery['id']);
    
    // Process the callback data
    if ($data == 'action:1') {
        $bot->editMessageText($chatId, $messageId, 'You selected Button 1');
    } elseif ($data == 'action:2') {
        $bot->editMessageText($chatId, $messageId, 'You selected Button 2');
    }
}
```

### Broadcasting a Message to All Users

```php
$bot = new TelegramBot(BOT_TOKEN, $db);

$result = $bot->broadcastMessage('Important announcement for all users!');

echo "Message sent to {$result['success']} users, failed for {$result['fail']} users.";
```

### Setting a Webhook

```php
$bot = new TelegramBot(BOT_TOKEN, $db);

$webhookUrl = 'https://your-domain.com/webhook.php';
$result = $bot->setWebhook($webhookUrl);

if ($result && $result['ok']) {
    echo "Webhook set successfully!";
} else {
    echo "Failed to set webhook.";
}
```

## Error Handling

The TelegramBot class uses exceptions for error handling. It's recommended to wrap API calls in try-catch blocks:

```php
try {
    $bot = new TelegramBot(BOT_TOKEN, $db);
    $bot->sendMessage(123456789, 'Hello, world!');
} catch (Exception $e) {
    // Log the error
    error_log("Telegram Bot Error: " . $e->getMessage());
    
    // Handle the error
    // ...
}
```

## Best Practices

1. **Rate Limiting**: Respect Telegram's rate limits (30 messages per second)
2. **Error Handling**: Always handle potential errors from API calls
3. **User Blocking**: Implement user blocking for spam prevention
4. **Logging**: Log important events and errors for debugging
5. **Security**: Keep your bot token secure and never expose it in client-side code

## References

- [Telegram Bot API Documentation](https://core.telegram.org/bots/api)
- [PHP Documentation](https://www.php.net/docs.php)