# Developer Guide for Telegram Bot

## Architecture Overview

This Telegram bot is built with PHP and uses a MySQL database for data storage. The architecture follows a modular approach with clear separation of concerns:

1. **Webhook Handler**: Entry point for Telegram updates
2. **Bot Class**: Core functionality for interacting with Telegram API
3. **Database Layer**: Data persistence and retrieval
4. **Admin Panel**: Web interface for managing bot content and users

## Directory Structure

```
new_bot/
├── admin/              # Admin panel files
│   ├── broadcast.php   # Send messages to all users
│   ├── content.php     # Manage content
│   ├── index.php       # Admin dashboard
│   ├── links.php       # Manage links
│   ├── login.php       # Admin authentication
│   ├── menus.php       # Manage menus
│   ├── settings.php    # Bot settings
│   ├── submenus.php    # Manage submenus
│   ├── users.php       # User management
│   └── user_activity.php # View user activity
├── assets/            # Static assets
│   ├── css/           # Stylesheets
│   ├── js/            # JavaScript files
│   └── img/           # Images
├── classes/           # PHP classes
│   └── TelegramBot.php # Bot functionality
├── config/            # Configuration files
│   └── config.php     # Main configuration
├── database/          # Database files
│   ├── init_db.php    # Database initialization
│   └── schema.sql     # Database schema
├── includes/          # Shared PHP files
│   ├── Database.php   # Database connection
│   ├── functions.php  # Helper functions
│   └── Logger.php     # Logging functionality
├── logs/              # Log files
├── webhook.php        # Webhook entry point
└── index.php          # Redirect to admin panel
```

## Core Components

### TelegramBot Class

The `TelegramBot` class handles all interactions with the Telegram Bot API. It provides methods for sending messages, handling updates, and managing users.

Key methods:
- `sendMessage()`: Send a message to a chat
- `editMessageText()`: Edit an existing message
- `processUpdate()`: Process an update from Telegram
- `saveUser()`: Save user information to the database

### Database Structure

The database consists of several tables:

1. **users**: Stores user information
2. **admins**: Admin user credentials
3. **sessions**: Admin session data
4. **menus**: Main menu categories
5. **submenus**: Submenus under main categories
6. **content**: Content items under submenus
7. **links**: External links under submenus
8. **broadcasts**: Message broadcast history
9. **activity_logs**: User activity tracking
10. **settings**: Bot configuration settings

### Webhook Handler

The `webhook.php` file receives updates from Telegram and passes them to the `TelegramBot` class for processing. It handles:

- Message updates
- Callback query updates (inline keyboard interactions)
- Error handling and logging

## Extending the Bot

### Adding New Commands

To add a new command:

1. Open `includes/functions.php`
2. Find the `processCommand()` function
3. Add a new case to the switch statement:

```php
case '/newcommand':
    // Handle the new command
    $bot->sendMessage($chatId, 'This is a new command!');
    break;
```

### Adding New Menu Types

To add a new menu type:

1. Create a new table in the database
2. Add functions to handle the new menu type in `includes/functions.php`
3. Update the admin panel to manage the new menu type
4. Modify the callback query processing to handle the new menu type

### Custom Keyboard Layouts

To create custom keyboard layouts:

1. Use the `inlineKeyboard()` function in `includes/functions.php`
2. Arrange buttons in rows and columns as needed

Example:

```php
$buttons = [
    [['text' => 'Button 1', 'callback_data' => 'action:1']],
    [
        ['text' => 'Button 2', 'callback_data' => 'action:2'],
        ['text' => 'Button 3', 'callback_data' => 'action:3']
    ]
];

$keyboard = inlineKeyboard($buttons);
```

## Admin Panel

### Authentication

The admin panel uses a simple session-based authentication system. Admin credentials are stored in the `admins` table.

### Content Management

The admin panel provides interfaces for managing:

1. **Menus**: Main categories
2. **Submenus**: Subcategories under main menus
3. **Content**: Text content under submenus
4. **Links**: External links under submenus

### User Management

Admins can:

1. View user list
2. Block/unblock users
3. Send messages to individual users
4. View user activity

### Broadcasting

The broadcast feature allows sending messages to all users or filtered groups of users.

## Webhook Setup

The bot requires a valid HTTPS webhook to receive updates from Telegram. The webhook can be set up through:

1. The admin panel (Settings page)
2. Direct API call to Telegram

```
https://api.telegram.org/bot<BOT_TOKEN>/setWebhook?url=https://your-domain.com/path/to/webhook.php
```

## Error Handling and Logging

The bot uses a custom `Logger` class for logging events and errors. Logs are stored in the `logs/` directory.

- `bot.log`: General bot activity
- `error.log`: Errors and exceptions

## Security Considerations

1. **Bot Token**: Keep your bot token secure and never expose it in client-side code
2. **Admin Credentials**: Use strong passwords for admin accounts
3. **Input Validation**: All user input is validated before processing
4. **Database Security**: Prepared statements are used to prevent SQL injection
5. **Error Handling**: Errors are logged but not exposed to users

## Performance Optimization

1. **Database Indexing**: Key fields are indexed for faster queries
2. **Caching**: Frequently accessed data can be cached
3. **Batch Processing**: For broadcasting to many users, use batch processing

## Testing

To test the bot:

1. Set up a test bot using BotFather
2. Configure the test bot in a development environment
3. Test all features manually
4. Create automated tests for critical functionality

## Deployment

1. Upload files to your web server
2. Set up the database using `database/init_db.php`
3. Configure the bot in `config/config.php`
4. Set up the webhook
5. Test the bot in production

## Troubleshooting

### Common Issues

1. **Webhook Not Working**:
   - Ensure your server has a valid SSL certificate
   - Check that the webhook URL is correct and accessible
   - Verify that the `webhook.php` file has the correct permissions

2. **Database Connection Issues**:
   - Verify your database credentials in `config/config.php`
   - Ensure the database user has the necessary permissions
   - Check if the database server is running

3. **Bot Not Responding**:
   - Verify that the bot token is correct
   - Check if the webhook is set correctly
   - Look for errors in the logs

### Debugging

To enable debug mode:

1. Set `DEBUG_MODE` to `true` in `config/config.php`
2. Check the logs for detailed information

## Contributing

When contributing to this project:

1. Follow the existing code style
2. Document your changes
3. Test thoroughly before submitting
4. Create clear, focused pull requests

## Resources

- [Telegram Bot API Documentation](https://core.telegram.org/bots/api)
- [PHP Documentation](https://www.php.net/docs.php)
- [MySQL Documentation](https://dev.mysql.com/doc/)