# Troubleshooting Guide for Telegram Bot

## Common Issues and Solutions

### Webhook Issues

#### Problem: Webhook Setup Fails

**Symptoms:**
- Error message when setting webhook
- Bot not receiving updates
- Error in Telegram API response

**Possible Causes:**
1. Invalid SSL certificate
2. Server not accessible via HTTPS
3. Incorrect webhook URL
4. Server firewall blocking Telegram

**Solutions:**
1. **Check SSL Certificate**
   - Ensure you have a valid SSL certificate (not self-signed)
   - Verify certificate is properly installed
   - Test your SSL configuration with [SSL Labs](https://www.ssllabs.com/ssltest/)

2. **Verify Server Accessibility**
   - Confirm your server is accessible from the internet
   - Try accessing the webhook URL in a browser
   - Check server logs for connection attempts

3. **Validate Webhook URL**
   - Ensure the URL is correct and complete (https://your-domain.com/path/to/webhook.php)
   - Check for typos in the domain or path
   - Verify the webhook file exists at the specified path

4. **Check Webhook Status**
   ```
   https://api.telegram.org/bot<YOUR_BOT_TOKEN>/getWebhookInfo
   ```
   - Look for errors in the response
   - Verify the URL matches your intended webhook URL

5. **Check Server Firewall**
   - Ensure your firewall allows incoming connections on port 443
   - Verify that Telegram's IP addresses are not blocked

#### Problem: Webhook Works Intermittently

**Symptoms:**
- Bot responds to some messages but not others
- Delayed responses
- Inconsistent behavior

**Possible Causes:**
1. Server resource limitations
2. PHP execution timeouts
3. Database connection issues

**Solutions:**
1. **Check Server Resources**
   - Monitor CPU and memory usage
   - Consider upgrading your hosting plan if resources are limited

2. **Adjust PHP Settings**
   - Increase max_execution_time in php.ini
   - Optimize database queries

3. **Enable Detailed Logging**
   - Set DEBUG_MODE to true in config.php
   - Check logs for errors or timeouts

### Database Issues

#### Problem: Database Connection Fails

**Symptoms:**
- Error messages about database connection
- Bot not responding or returning errors
- Admin panel not loading

**Possible Causes:**
1. Incorrect database credentials
2. Database server down
3. Connection limits reached

**Solutions:**
1. **Verify Database Credentials**
   - Check DB_HOST, DB_NAME, DB_USER, and DB_PASS in config.php
   - Try connecting to the database manually

2. **Check Database Server Status**
   - Ensure the database server is running
   - Check for any maintenance or outages

3. **Test Database Connection**
   - Create a simple PHP script to test the connection:
   ```php
   <?php
   try {
       $dsn = "mysql:host=localhost;dbname=your_db_name;charset=utf8mb4";
       $pdo = new PDO($dsn, 'your_db_user', 'your_db_pass');
       echo "Connection successful!";
   } catch (PDOException $e) {
       echo "Connection failed: " . $e->getMessage();
   }
   ```

#### Problem: Database Queries Failing

**Symptoms:**
- Error messages in logs
- Incomplete data in admin panel
- Bot not responding correctly

**Possible Causes:**
1. Missing or incorrect tables
2. SQL syntax errors
3. Permission issues

**Solutions:**
1. **Verify Database Schema**
   - Check if all required tables exist
   - Run the database initialization script again
   ```
   php database/init_db.php
   ```

2. **Check Table Permissions**
   - Ensure the database user has SELECT, INSERT, UPDATE, and DELETE permissions

3. **Enable Query Logging**
   - Temporarily enable MySQL query logging
   - Check for SQL syntax errors

### Bot Response Issues

#### Problem: Bot Not Responding to Commands

**Symptoms:**
- Bot doesn't respond to /start or other commands
- No error messages
- Webhook is set up correctly

**Possible Causes:**
1. Command handling code issues
2. Bot token incorrect
3. Bot blocked by user

**Solutions:**
1. **Verify Bot Token**
   - Check BOT_TOKEN in config.php
   - Test the token with a simple API request:
   ```
   https://api.telegram.org/bot<YOUR_BOT_TOKEN>/getMe
   ```

2. **Check Command Handling**
   - Review the processCommand function in functions.php
   - Add debug logging to track command processing

3. **Test Bot Status**
   - Start a new conversation with the bot
   - Check if the bot appears online

4. **Review Logs**
   - Check bot.log and error.log for issues
   - Enable more detailed logging temporarily

#### Problem: Formatting Issues in Messages

**Symptoms:**
- Messages display incorrectly
- Markdown or HTML not rendering
- Escaped characters in text

**Possible Causes:**
1. Incorrect parse_mode
2. Invalid Markdown or HTML syntax
3. Special characters not escaped properly

**Solutions:**
1. **Check Parse Mode**
   - Ensure you're using the correct parse_mode ('Markdown' or 'HTML')
   - Be consistent with the chosen format

2. **Validate Message Format**
   - For Markdown, escape special characters: _ * [ ] ( ) ~ ` > #
   - For HTML, use only supported tags: <b>, <i>, <a>, <code>, <pre>

3. **Use the formatMessageText Function**
   - Utilize the built-in formatMessageText function for Markdown

### Admin Panel Issues

#### Problem: Cannot Log In to Admin Panel

**Symptoms:**
- Login attempts fail
- Redirect loop
- Blank page after login

**Possible Causes:**
1. Incorrect credentials
2. Session issues
3. PHP configuration problems

**Solutions:**
1. **Verify Admin Credentials**
   - Check ADMIN_PASSWORD in config.php
   - Reset the password in the database if necessary

2. **Check Session Configuration**
   - Ensure PHP sessions are working
   - Verify session directory is writable

3. **Clear Browser Cache and Cookies**
   - Try logging in from a different browser
   - Clear cookies and cache

4. **Check PHP Error Logs**
   - Look for session-related errors
   - Verify PHP version compatibility

#### Problem: Admin Panel Features Not Working

**Symptoms:**
- Cannot add/edit/delete items
- Forms submit but no changes occur
- JavaScript errors in browser console

**Possible Causes:**
1. JavaScript errors
2. AJAX request issues
3. Permission problems

**Solutions:**
1. **Check Browser Console**
   - Open browser developer tools (F12)
   - Look for JavaScript errors

2. **Verify File Permissions**
   - Ensure PHP has write access to the database

3. **Test Form Submission**
   - Try disabling JavaScript and submit forms directly
   - Check network requests in browser developer tools

4. **Review JavaScript Files**
   - Check for syntax errors in JS files
   - Verify jQuery and Bootstrap are loading correctly

### Content Display Issues

#### Problem: Content Not Displaying in Bot

**Symptoms:**
- Empty messages
- Missing menus or submenus
- Error messages instead of content

**Possible Causes:**
1. Missing content in database
2. Formatting issues
3. Callback data processing errors

**Solutions:**
1. **Check Database Content**
   - Verify menus, submenus, and content exist in the database
   - Check for empty fields

2. **Review Callback Processing**
   - Check the processCallbackQuery function in functions.php
   - Add debug logging to track callback processing

3. **Test Content Display Functions**
   - Review showMenu, showSubmenu, and showContent functions
   - Add temporary debug messages

#### Problem: Images or Documents Not Sending

**Symptoms:**
- Error messages when trying to send media
- Media not appearing in messages
- Timeout errors

**Possible Causes:**
1. File size too large
2. Invalid file path
3. Unsupported file format

**Solutions:**
1. **Check File Size**
   - Telegram limits: 10MB for photos, 50MB for other files
   - Resize images if necessary

2. **Verify File Paths**
   - Ensure file paths are correct and accessible
   - Check file permissions

3. **Test with Known Working Files**
   - Try sending a small JPEG image
   - Use file_id for previously uploaded files

### Performance Issues

#### Problem: Slow Response Times

**Symptoms:**
- Bot takes a long time to respond
- Admin panel loads slowly
- Timeouts during operations

**Possible Causes:**
1. Server resource limitations
2. Inefficient database queries
3. Large data sets

**Solutions:**
1. **Optimize Database Queries**
   - Add indexes to frequently queried columns
   - Limit result sets
   - Use pagination for large data sets

2. **Implement Caching**
   - Cache frequently accessed data
   - Use memcached or Redis if available

3. **Monitor Server Resources**
   - Check CPU, memory, and disk usage
   - Consider upgrading hosting if necessary

4. **Enable PHP OpCache**
   - Configure PHP OpCache for better performance
   - Adjust PHP memory limits if needed

## Diagnostic Tools

### Webhook Tester

Create a simple webhook tester script to verify your server can receive webhook requests:

```php
<?php
// webhook_test.php
$content = file_get_contents('php://input');
$time = date('Y-m-d H:i:s');
file_put_contents('webhook_test.log', "[$time] Request received: $content\n", FILE_APPEND);
echo "OK";
```

Set this as your webhook temporarily to test if Telegram can reach your server.

### Database Connection Tester

Create a database connection test script:

```php
<?php
// db_test.php
require_once 'config/config.php';

try {
    $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    $db = new PDO($dsn, DB_USER, DB_PASS, $options);
    
    echo "Database connection successful!<br>";
    
    // Test tables
    $tables = ['users', 'menus', 'submenus', 'content', 'links', 'settings'];
    foreach ($tables as $table) {
        $stmt = $db->query("SELECT COUNT(*) FROM $table");
        $count = $stmt->fetchColumn();
        echo "Table $table: $count records<br>";
    }
    
} catch (PDOException $e) {
    echo "Database connection failed: " . $e->getMessage();
}
```

### Log Viewer

Create a simple log viewer for the admin panel:

```php
<?php
// admin/logs.php
require_once '../includes/auth.php'; // Ensure admin is logged in

$logFile = isset($_GET['type']) && $_GET['type'] == 'error' ? '../logs/error.log' : '../logs/bot.log';
$lines = isset($_GET['lines']) ? (int)$_GET['lines'] : 100;

if (file_exists($logFile)) {
    $logs = file($logFile);
    $logs = array_slice($logs, -$lines);
    $logContent = implode('', $logs);
} else {
    $logContent = "Log file not found.";
}

// Display with syntax highlighting
$logContent = htmlspecialchars($logContent);
?>
<!DOCTYPE html>
<html>
<head>
    <title>Log Viewer</title>
    <style>
        body { font-family: monospace; background: #f5f5f5; }
        .log-container { background: #fff; padding: 20px; border-radius: 5px; box-shadow: 0 2px 5px rgba(0,0,0,0.1); }
        .error { color: #d9534f; }
        .warning { color: #f0ad4e; }
        .info { color: #5bc0de; }
    </style>
</head>
<body>
    <h1>Log Viewer</h1>
    <p>
        <a href="?type=bot">Bot Log</a> | 
        <a href="?type=error">Error Log</a> | 
        Show lines: 
        <a href="?type=<?= isset($_GET['type']) ? $_GET['type'] : 'bot' ?>&lines=50">50</a> | 
        <a href="?type=<?= isset($_GET['type']) ? $_GET['type'] : 'bot' ?>&lines=100">100</a> | 
        <a href="?type=<?= isset($_GET['type']) ? $_GET['type'] : 'bot' ?>&lines=500">500</a>
    </p>
    <div class="log-container">
        <pre><?= $logContent ?></pre>
    </div>
</body>
</html>
```

## Advanced Troubleshooting

### Enabling Debug Mode

To enable detailed debugging:

1. Edit `config/config.php`
2. Add or modify the DEBUG_MODE constant:
   ```php
   define('DEBUG_MODE', true);
   ```
3. Restart the web server if necessary
4. Check the logs for detailed information

### Database Repair

If you suspect database corruption:

1. Back up your database
2. Run repair commands:
   ```sql
   REPAIR TABLE users, menus, submenus, content, links, settings;
   ```
3. Check for and fix foreign key constraints:
   ```sql
   SET FOREIGN_KEY_CHECKS=0;
   -- Run your repairs or modifications
   SET FOREIGN_KEY_CHECKS=1;
   ```

### Reinstalling the Bot

As a last resort, you can reinstall the bot while preserving data:

1. Back up your database
2. Back up your `config/config.php` file
3. Re-upload the bot files
4. Restore your configuration file
5. Run database updates if necessary

## Getting Help

If you've tried the solutions in this guide and still have issues:

1. Check the project documentation
2. Search for similar issues in the project repository
3. Provide the following information when seeking help:
   - Detailed description of the issue
   - Steps to reproduce
   - Error messages and logs
   - PHP and MySQL versions
   - Server environment details