<?php
/**
 * Admin Authentication Handler
 * This file handles authentication for admin panel pages
 */

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Include required files if not already included
if (!defined('ADMIN_PATH')) {
    require_once dirname(__DIR__) . '/config/config.php';
}

if (!isset($db)) {
    require_once dirname(__DIR__) . '/includes/database.php';
}

/**
 * Check if admin is logged in
 * @return bool True if logged in, false otherwise
 */
function isAdminLoggedIn() {
    if (!isset($_SESSION['admin_id']) || !isset($_SESSION['admin_token'])) {
        return false;
    }
    
    // Verify session token in database
    global $db;
    
    try {
        $stmt = $db->prepare("SELECT * FROM admin_sessions 
                            WHERE admin_id = ? AND session_token = ? AND expires_at > NOW()");
        $stmt->execute([$_SESSION['admin_id'], $_SESSION['admin_token']]);
        $session = $stmt->fetch();
        
        if (!$session) {
            // Invalid or expired session
            return false;
        }
        
        // Extend session if needed
        if (strtotime($session['expires_at']) - time() < SESSION_LIFETIME / 2) {
            // Session is halfway to expiration, extend it
            $stmt = $db->prepare("UPDATE admin_sessions 
                                SET expires_at = DATE_ADD(NOW(), INTERVAL ? SECOND) 
                                WHERE admin_id = ? AND session_token = ?");
            $stmt->execute([SESSION_LIFETIME, $_SESSION['admin_id'], $_SESSION['admin_token']]);
        }
        
        return true;
    } catch (Exception $e) {
        logError("Session verification error: " . $e->getMessage());
        return false;
    }
}

/**
 * Get current admin user data
 * @return array|bool Admin data or false if not logged in
 */
function getCurrentAdmin() {
    if (!isAdminLoggedIn()) {
        return false;
    }
    
    global $db;
    
    try {
        $stmt = $db->prepare("SELECT id, username, email, last_login FROM admin_users WHERE id = ?");
        $stmt->execute([$_SESSION['admin_id']]);
        return $stmt->fetch();
    } catch (Exception $e) {
        logError("Get admin error: " . $e->getMessage());
        return false;
    }
}

// Check if admin is logged in, redirect to login page if not
if (!isAdminLoggedIn()) {
    // Store current URL for redirect after login
    $_SESSION['redirect_after_login'] = $_SERVER['REQUEST_URI'];
    
    // Redirect to login page
    header('Location: login.php');
    exit;
}

// Get current admin data
$currentAdmin = getCurrentAdmin();

// Update last login time if needed
if ($currentAdmin && !isset($_SESSION['last_login_updated'])) {
    try {
        $stmt = $db->prepare("UPDATE admin_users SET last_login = NOW() WHERE id = ?");
        $stmt->execute([$currentAdmin['id']]);
        $_SESSION['last_login_updated'] = true;
    } catch (Exception $e) {
        logError("Update last login error: " . $e->getMessage());
    }
}