<?php
/**
 * Broadcast Messages
 */

// Include authentication
require_once 'auth.php';

// Initialize variables
$error = '';
$success = '';
$broadcasts = [];

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Check if action is broadcast
    if (isset($_POST['action']) && $_POST['action'] === 'broadcast') {
        $message = isset($_POST['message']) ? trim($_POST['message']) : '';
        
        if (empty($message)) {
            $error = 'Message cannot be empty.';
        } else {
            // Get all active users
            try {
                $stmt = $db->prepare("SELECT * FROM users WHERE status = 'active'");
                $stmt->execute();
                $users = $stmt->fetchAll();
                
                if (empty($users)) {
                    $error = 'No active users found to send broadcast to.';
                } else {
                    // Initialize TelegramBot class
                    require_once '../classes/TelegramBot.php';
                    $bot = new TelegramBot(BOT_TOKEN);
                    
                    // Track success and failure counts
                    $successCount = 0;
                    $failCount = 0;
                    
                    // Send message to each user
                    foreach ($users as $user) {
                        $result = $bot->sendMessage($user['telegram_id'], $message);
                        
                        if ($result) {
                            $successCount++;
                        } else {
                            $failCount++;
                        }
                        
                        // Add a small delay to avoid hitting Telegram API limits
                        usleep(50000); // 50ms delay
                    }
                    
                    // Log the broadcast
                    try {
                        $stmt = $db->prepare("INSERT INTO broadcasts 
                                            (admin_id, message, sent_count, failed_count) 
                                            VALUES (?, ?, ?, ?)");
                        $stmt->execute([$_SESSION['admin_id'], $message, $successCount, $failCount]);
                        
                        // Log admin activity
                        logActivity('admin', 'broadcast', "Sent broadcast message to {$successCount} users", $_SESSION['admin_id']);
                        
                        $success = "Broadcast sent successfully to {$successCount} users" . 
                                  ($failCount > 0 ? " ({$failCount} failed)" : '');
                    } catch (Exception $e) {
                        logError("Broadcast log error: " . $e->getMessage());
                        $error = 'Error logging broadcast. Please try again.';
                    }
                }
            } catch (Exception $e) {
                logError("Broadcast users error: " . $e->getMessage());
                $error = 'Error retrieving users. Please try again.';
            }
        }
    }
}

// Get recent broadcasts
try {
    $stmt = $db->prepare("SELECT b.*, a.username as admin_username 
                        FROM broadcasts b 
                        LEFT JOIN admins a ON b.admin_id = a.id 
                        ORDER BY b.created_at DESC 
                        LIMIT 20");
    $stmt->execute();
    $broadcasts = $stmt->fetchAll();
} catch (Exception $e) {
    logError("Load broadcasts error: " . $e->getMessage());
    $error = 'Error loading broadcast history. Please try again.';
}

// Page title
$pageTitle = "Broadcast Messages";

// Include header
include 'includes/header.php';
?>

<div class="container-fluid py-4">
    <h1 class="h3 mb-4"><?php echo $pageTitle; ?></h1>
    
    <?php if ($error): ?>
        <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <?php if ($success): ?>
        <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
    <?php endif; ?>
    
    <div class="row">
        <div class="col-lg-6">
            <!-- New Broadcast Form -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Send New Broadcast</h6>
                </div>
                <div class="card-body">
                    <form method="post" action="">
                        <input type="hidden" name="action" value="broadcast">
                        
                        <div class="mb-3">
                            <label for="message" class="form-label">Message</label>
                            <textarea class="form-control" id="message" name="message" rows="6" required></textarea>
                            <small class="text-muted">
                                This message will be sent to all active users. You can use Telegram's supported HTML formatting.
                            </small>
                        </div>
                        
                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-paper-plane"></i> Send Broadcast
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <div class="col-lg-6">
            <!-- Broadcast Stats -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Broadcast Statistics</h6>
                </div>
                <div class="card-body">
                    <?php
                    // Get user counts
                    $totalUsers = 0;
                    $activeUsers = 0;
                    
                    try {
                        $stmt = $db->query("SELECT COUNT(*) FROM users");
                        $totalUsers = $stmt->fetchColumn();
                        
                        $stmt = $db->query("SELECT COUNT(*) FROM users WHERE status = 'active'");
                        $activeUsers = $stmt->fetchColumn();
                    } catch (Exception $e) {
                        logError("User count error: " . $e->getMessage());
                    }
                    ?>
                    
                    <div class="row">
                        <div class="col-md-6 mb-4">
                            <div class="card border-left-primary shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">
                                                Total Users</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $totalUsers; ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-users fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6 mb-4">
                            <div class="card border-left-success shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                                Active Users</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $activeUsers; ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-user-check fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle"></i> Your broadcast will be sent to <strong><?php echo $activeUsers; ?></strong> active users.
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Broadcast History -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">Broadcast History</h6>
        </div>
        <div class="card-body">
            <?php if (empty($broadcasts)): ?>
                <p class="text-center text-muted">No broadcast history found.</p>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-bordered table-hover" id="broadcastsTable" width="100%" cellspacing="0">
                        <thead>
                            <tr>
                                <th>Date/Time</th>
                                <th>Admin</th>
                                <th>Message</th>
                                <th>Sent</th>
                                <th>Failed</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($broadcasts as $broadcast): ?>
                                <tr>
                                    <td><?php echo date('M d, Y H:i', strtotime($broadcast['created_at'])); ?></td>
                                    <td><?php echo htmlspecialchars($broadcast['admin_username']); ?></td>
                                    <td>
                                        <div class="text-truncate" style="max-width: 300px;">
                                            <?php echo htmlspecialchars($broadcast['message']); ?>
                                        </div>
                                    </td>
                                    <td><?php echo $broadcast['sent_count']; ?></td>
                                    <td>
                                        <?php if ($broadcast['failed_count'] > 0): ?>
                                            <span class="text-danger"><?php echo $broadcast['failed_count']; ?></span>
                                        <?php else: ?>
                                            0
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php
// Add page-specific scripts
$pageScripts = ['../assets/js/broadcast.js'];

// Include footer
include 'includes/footer.php';
?>