<?php
/**
 * Content Management
 */

// Include authentication
require_once 'auth.php';

// Initialize variables
$error = '';
$success = '';
$content = [];
$submenus = [];
$selectedSubmenu = null;

// Get filter parameters
$submenuId = isset($_GET['submenu_id']) ? (int)$_GET['submenu_id'] : 0;

// Process form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Check action type
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add') {
        // Add new content
        $submenuId = (int)($_POST['submenu_id'] ?? 0);
        $title = trim($_POST['title'] ?? '');
        $contentText = trim($_POST['content'] ?? '');
        $order = (int)($_POST['order'] ?? 0);
        $status = $_POST['status'] ?? 'active';
        
        if (empty($title) || $submenuId <= 0) {
            $error = 'Submenu and content title are required.';
        } else {
            try {
                $stmt = $db->prepare("INSERT INTO content 
                                    (submenu_id, title, content, display_order, status, created_by) 
                                    VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->execute([$submenuId, $title, $contentText, $order, $status, $_SESSION['admin_id']]);
                
                $success = 'Content added successfully.';
                logActivity('admin', null, $_SESSION['admin_id'], 'content_add', "Added new content: $title", $_SERVER['REMOTE_ADDR']);
            } catch (Exception $e) {
                $error = 'Error adding content. Please try again.';
                logError("Add content error: " . $e->getMessage());
            }
        }
    } elseif ($action === 'edit') {
        // Edit existing content
        $id = (int)($_POST['id'] ?? 0);
        $submenuId = (int)($_POST['submenu_id'] ?? 0);
        $title = trim($_POST['title'] ?? '');
        $contentText = trim($_POST['content'] ?? '');
        $order = (int)($_POST['order'] ?? 0);
        $status = $_POST['status'] ?? 'active';
        
        if (empty($title) || $id <= 0 || $submenuId <= 0) {
            $error = 'Invalid content data.';
        } else {
            try {
                $stmt = $db->prepare("UPDATE content 
                                    SET submenu_id = ?, title = ?, content = ?, 
                                        display_order = ?, status = ?, updated_at = NOW() 
                                    WHERE id = ?");
                $stmt->execute([$submenuId, $title, $contentText, $order, $status, $id]);
                
                $success = 'Content updated successfully.';
                logActivity('admin', null, $_SESSION['admin_id'], 'content_edit', "Updated content ID: $id", $_SERVER['REMOTE_ADDR']);
            } catch (Exception $e) {
                $error = 'Error updating content. Please try again.';
                logError("Edit content error: " . $e->getMessage());
            }
        }
    } elseif ($action === 'delete') {
        // Delete content
        $id = (int)($_POST['id'] ?? 0);
        
        if ($id <= 0) {
            $error = 'Invalid content ID.';
        } else {
            try {
                $stmt = $db->prepare("DELETE FROM content WHERE id = ?");
                $stmt->execute([$id]);
                
                $success = 'Content deleted successfully.';
                logActivity('admin', null, $_SESSION['admin_id'], 'content_delete', "Deleted content ID: $id", $_SERVER['REMOTE_ADDR']);
            } catch (Exception $e) {
                $error = 'Error deleting content. Please try again.';
                logError("Delete content error: " . $e->getMessage());
            }
        }
    }
}

// Get all submenus for dropdown
try {
    $stmt = $db->query("SELECT s.id, s.title, m.title as menu_title 
                        FROM submenus s 
                        JOIN menus m ON s.menu_id = m.id 
                        WHERE s.status = 'active' 
                        ORDER BY m.display_order, m.title, s.display_order, s.title");
    $submenus = $stmt->fetchAll();
    
    // Get selected submenu details if submenu_id is provided
    if ($submenuId > 0) {
        $stmt = $db->prepare("SELECT s.id, s.title, m.title as menu_title 
                            FROM submenus s 
                            JOIN menus m ON s.menu_id = m.id 
                            WHERE s.id = ?");
        $stmt->execute([$submenuId]);
        $selectedSubmenu = $stmt->fetch();
    }
} catch (Exception $e) {
    $error = 'Error loading submenus. Please try again.';
    logError("Load submenus error: " . $e->getMessage());
}

// Get content based on filter
try {
    $query = "SELECT c.*, 
              s.title as submenu_title, m.title as menu_title,
              a.username as created_by_name
              FROM content c 
              LEFT JOIN submenus s ON c.submenu_id = s.id 
              LEFT JOIN menus m ON s.menu_id = m.id 
              LEFT JOIN admin_users a ON c.created_by = a.id ";
    
    $params = [];
    
    if ($submenuId > 0) {
        $query .= "WHERE c.submenu_id = ? ";
        $params[] = $submenuId;
    }
    
    $query .= "ORDER BY c.submenu_id, c.display_order, c.title";
    
    if (!empty($params)) {
        $stmt = $db->prepare($query);
        $stmt->execute($params);
    } else {
        $stmt = $db->query($query);
    }
    
    $content = $stmt->fetchAll();
} catch (Exception $e) {
    $error = 'Error loading content. Please try again.';
    logError("Load content error: " . $e->getMessage());
}

// Page title
$pageTitle = $selectedSubmenu ? "Content for {$selectedSubmenu['title']}" : "Content Management";

// Include header
include 'includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3"><?php echo htmlspecialchars($pageTitle); ?></h1>
            <?php if ($selectedSubmenu): ?>
                <a href="content.php" class="btn btn-sm btn-outline-secondary">
                    <i class="fas fa-arrow-left"></i> Show All Content
                </a>
            <?php endif; ?>
        </div>
        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addContentModal">
            <i class="fas fa-plus"></i> Add New Content
        </button>
    </div>
    
    <?php if ($error): ?>
        <div class="alert alert-danger alert-auto-dismiss"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <?php if ($success): ?>
        <div class="alert alert-success alert-auto-dismiss"><?php echo htmlspecialchars($success); ?></div>
    <?php endif; ?>
    
    <!-- Filter Form -->
    <?php if (!$selectedSubmenu): ?>
        <div class="card shadow mb-4">
            <div class="card-header py-3">
                <h6 class="m-0 font-weight-bold text-primary">Filter Content</h6>
            </div>
            <div class="card-body">
                <form method="get" action="" class="row g-3">
                    <div class="col-md-4">
                        <label for="submenu_id" class="form-label">Submenu</label>
                        <select class="form-select" id="submenu_id" name="submenu_id">
                            <option value="">All Submenus</option>
                            <?php foreach ($submenus as $submenu): ?>
                                <option value="<?php echo $submenu['id']; ?>">
                                    <?php echo htmlspecialchars($submenu['menu_title'] . ' > ' . $submenu['title']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary">Filter</button>
                    </div>
                </form>
            </div>
        </div>
    <?php endif; ?>
    
    <!-- Content Table -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">
                <?php echo $selectedSubmenu ? "Content for {$selectedSubmenu['title']}" : "All Content"; ?>
            </h6>
        </div>
        <div class="card-body">
            <?php if (empty($content)): ?>
                <p class="text-center text-muted">No content found.</p>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-bordered table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Menu > Submenu</th>
                                <th>Title</th>
                                <th>Content Preview</th>
                                <th>Order</th>
                                <th>Status</th>
                                <th>Created By</th>
                                <th>Created At</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($content as $item): ?>
                                <tr>
                                    <td><?php echo $item['id']; ?></td>
                                    <td><?php echo htmlspecialchars($item['menu_title'] . ' > ' . $item['submenu_title']); ?></td>
                                    <td><?php echo htmlspecialchars($item['title']); ?></td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-outline-info view-content"
                                                data-title="<?php echo htmlspecialchars($item['title']); ?>"
                                                data-content="<?php echo htmlspecialchars($item['content']); ?>"
                                                data-bs-toggle="modal" data-bs-target="#viewContentModal">
                                            Preview
                                        </button>
                                    </td>
                                    <td><?php echo $item['display_order']; ?></td>
                                    <td>
                                        <?php if ($item['status'] === 'active'): ?>
                                            <span class="badge bg-success">Active</span>
                                        <?php else: ?>
                                            <span class="badge bg-secondary">Inactive</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($item['created_by_name'] ?? 'System'); ?></td>
                                    <td><?php echo date('M d, Y H:i', strtotime($item['created_at'])); ?></td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-primary edit-content" 
                                                data-id="<?php echo $item['id']; ?>"
                                                data-submenu-id="<?php echo $item['submenu_id']; ?>"
                                                data-title="<?php echo htmlspecialchars($item['title']); ?>"
                                                data-content="<?php echo htmlspecialchars($item['content']); ?>"
                                                data-order="<?php echo $item['display_order']; ?>"
                                                data-status="<?php echo $item['status']; ?>"
                                                data-bs-toggle="modal" data-bs-target="#editContentModal">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button type="button" class="btn btn-sm btn-danger delete-content"
                                                data-id="<?php echo $item['id']; ?>"
                                                data-title="<?php echo htmlspecialchars($item['title']); ?>"
                                                data-bs-toggle="modal" data-bs-target="#deleteContentModal">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Add Content Modal -->
<div class="modal fade" id="addContentModal" tabindex="-1" aria-labelledby="addContentModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addContentModalLabel">Add New Content</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="post" action="">
                <div class="modal-body">
                    <input type="hidden" name="action" value="add">
                    
                    <div class="mb-3">
                        <label for="submenu_id" class="form-label">Submenu</label>
                        <select class="form-select" id="submenu_id" name="submenu_id" required>
                            <option value="">Select Submenu</option>
                            <?php foreach ($submenus as $submenu): ?>
                                <option value="<?php echo $submenu['id']; ?>" <?php echo ($submenuId == $submenu['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($submenu['menu_title'] . ' > ' . $submenu['title']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="title" class="form-label">Content Title</label>
                        <input type="text" class="form-control" id="title" name="title" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="content" class="form-label">Content</label>
                        <textarea class="form-control" id="content" name="content" rows="6"></textarea>
                    </div>
                    
                    <div class="mb-3">
                        <label for="order" class="form-label">Display Order</label>
                        <input type="number" class="form-control" id="order" name="order" value="0" min="0">
                        <small class="text-muted">Lower numbers appear first</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="status" class="form-label">Status</label>
                        <select class="form-select" id="status" name="status">
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Add Content</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Content Modal -->
<div class="modal fade" id="editContentModal" tabindex="-1" aria-labelledby="editContentModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editContentModalLabel">Edit Content</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="post" action="">
                <div class="modal-body">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="id" id="edit-id">
                    
                    <div class="mb-3">
                        <label for="edit-submenu-id" class="form-label">Submenu</label>
                        <select class="form-select" id="edit-submenu-id" name="submenu_id" required>
                            <option value="">Select Submenu</option>
                            <?php foreach ($submenus as $submenu): ?>
                                <option value="<?php echo $submenu['id']; ?>">
                                    <?php echo htmlspecialchars($submenu['menu_title'] . ' > ' . $submenu['title']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-title" class="form-label">Content Title</label>
                        <input type="text" class="form-control" id="edit-title" name="title" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-content" class="form-label">Content</label>
                        <textarea class="form-control" id="edit-content" name="content" rows="6"></textarea>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-order" class="form-label">Display Order</label>
                        <input type="number" class="form-control" id="edit-order" name="order" value="0" min="0">
                        <small class="text-muted">Lower numbers appear first</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-status" class="form-label">Status</label>
                        <select class="form-select" id="edit-status" name="status">
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Update Content</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- View Content Modal -->
<div class="modal fade" id="viewContentModal" tabindex="-1" aria-labelledby="viewContentModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="viewContentModalLabel">Content Preview</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <h4 id="view-content-title"></h4>
                <hr>
                <div id="view-content-body" class="mt-3"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<!-- Delete Content Modal -->
<div class="modal fade" id="deleteContentModal" tabindex="-1" aria-labelledby="deleteContentModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteContentModalLabel">Delete Content</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete the content <strong id="delete-content-title"></strong>?</p>
                <p class="text-danger">This action cannot be undone.</p>
            </div>
            <form method="post" action="">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="id" id="delete-id">
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">Delete Content</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php
// Add page-specific scripts
$pageScripts = ['../assets/js/content.js'];

// Include footer
include 'includes/footer.php';
?>