<?php
/**
 * Links Management
 */

// Include authentication
require_once 'auth.php';

// Initialize variables
$error = '';
$success = '';
$links = [];
$submenus = [];
$selectedSubmenu = null;

// Get filter parameters
$submenuId = isset($_GET['submenu_id']) ? (int)$_GET['submenu_id'] : 0;
$type = isset($_GET['type']) ? $_GET['type'] : '';

// Process form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Check action type
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add') {
        // Add new link
        $submenuId = (int)($_POST['submenu_id'] ?? 0);
        $title = trim($_POST['title'] ?? '');
        $url = trim($_POST['url'] ?? '');
        $type = $_POST['type'] ?? 'website';
        $order = (int)($_POST['order'] ?? 0);
        $status = $_POST['status'] ?? 'active';
        
        if (empty($title) || empty($url) || $submenuId <= 0) {
            $error = 'Submenu, title and URL are required.';
        } else {
            try {
                $stmt = $db->prepare("INSERT INTO links 
                                    (submenu_id, title, url, type, display_order, status, created_by) 
                                    VALUES (?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$submenuId, $title, $url, $type, $order, $status, $_SESSION['admin_id']]);
                
                $success = 'Link added successfully.';
                logActivity('admin', null, $_SESSION['admin_id'], 'link_add', "Added new link: $title", $_SERVER['REMOTE_ADDR']);
            } catch (Exception $e) {
                $error = 'Error adding link. Please try again.';
                logError("Add link error: " . $e->getMessage());
            }
        }
    } elseif ($action === 'edit') {
        // Edit existing link
        $id = (int)($_POST['id'] ?? 0);
        $submenuId = (int)($_POST['submenu_id'] ?? 0);
        $title = trim($_POST['title'] ?? '');
        $url = trim($_POST['url'] ?? '');
        $type = $_POST['type'] ?? 'website';
        $order = (int)($_POST['order'] ?? 0);
        $status = $_POST['status'] ?? 'active';
        
        if (empty($title) || empty($url) || $id <= 0 || $submenuId <= 0) {
            $error = 'Invalid link data.';
        } else {
            try {
                $stmt = $db->prepare("UPDATE links 
                                    SET submenu_id = ?, title = ?, url = ?, 
                                        type = ?, display_order = ?, status = ?, updated_at = NOW() 
                                    WHERE id = ?");
                $stmt->execute([$submenuId, $title, $url, $type, $order, $status, $id]);
                
                $success = 'Link updated successfully.';
                logActivity('admin', null, $_SESSION['admin_id'], 'link_edit', "Updated link ID: $id", $_SERVER['REMOTE_ADDR']);
            } catch (Exception $e) {
                $error = 'Error updating link. Please try again.';
                logError("Edit link error: " . $e->getMessage());
            }
        }
    } elseif ($action === 'delete') {
        // Delete link
        $id = (int)($_POST['id'] ?? 0);
        
        if ($id <= 0) {
            $error = 'Invalid link ID.';
        } else {
            try {
                $stmt = $db->prepare("DELETE FROM links WHERE id = ?");
                $stmt->execute([$id]);
                
                $success = 'Link deleted successfully.';
                logActivity('admin', null, $_SESSION['admin_id'], 'link_delete', "Deleted link ID: $id", $_SERVER['REMOTE_ADDR']);
            } catch (Exception $e) {
                $error = 'Error deleting link. Please try again.';
                logError("Delete link error: " . $e->getMessage());
            }
        }
    }
}

// Get all submenus for dropdown
try {
    $stmt = $db->query("SELECT s.id, s.title, m.title as menu_title 
                        FROM submenus s 
                        JOIN menus m ON s.menu_id = m.id 
                        WHERE s.status = 'active' 
                        ORDER BY m.display_order, m.title, s.display_order, s.title");
    $submenus = $stmt->fetchAll();
    
    // Get selected submenu details if submenu_id is provided
    if ($submenuId > 0) {
        $stmt = $db->prepare("SELECT s.id, s.title, m.title as menu_title 
                            FROM submenus s 
                            JOIN menus m ON s.menu_id = m.id 
                            WHERE s.id = ?");
        $stmt->execute([$submenuId]);
        $selectedSubmenu = $stmt->fetch();
    }
} catch (Exception $e) {
    $error = 'Error loading submenus. Please try again.';
    logError("Load submenus error: " . $e->getMessage());
}

// Get links based on filter
try {
    $query = "SELECT l.*, 
              s.title as submenu_title, m.title as menu_title,
              a.username as created_by_name
              FROM links l 
              LEFT JOIN submenus s ON l.submenu_id = s.id 
              LEFT JOIN menus m ON s.menu_id = m.id 
              LEFT JOIN admin_users a ON l.created_by = a.id ";
    
    $params = [];
    $whereAdded = false;
    
    if ($submenuId > 0) {
        $query .= "WHERE l.submenu_id = ? ";
        $params[] = $submenuId;
        $whereAdded = true;
    }
    
    if (!empty($type)) {
        if ($whereAdded) {
            $query .= "AND l.type = ? ";
        } else {
            $query .= "WHERE l.type = ? ";
            $whereAdded = true;
        }
        $params[] = $type;
    }
    
    $query .= "ORDER BY l.submenu_id, l.type, l.display_order, l.title";
    
    if (!empty($params)) {
        $stmt = $db->prepare($query);
        $stmt->execute($params);
    } else {
        $stmt = $db->query($query);
    }
    
    $links = $stmt->fetchAll();
} catch (Exception $e) {
    $error = 'Error loading links. Please try again.';
    logError("Load links error: " . $e->getMessage());
}

// Page title
$pageTitle = "Links Management";
if ($selectedSubmenu) {
    $pageTitle = "Links for {$selectedSubmenu['title']}";
} elseif (!empty($type)) {
    $pageTitle = ucfirst($type) . " Links";
}

// Include header
include 'includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3"><?php echo htmlspecialchars($pageTitle); ?></h1>
            <?php if ($selectedSubmenu || !empty($type)): ?>
                <a href="links.php" class="btn btn-sm btn-outline-secondary">
                    <i class="fas fa-arrow-left"></i> Show All Links
                </a>
            <?php endif; ?>
        </div>
        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addLinkModal">
            <i class="fas fa-plus"></i> Add New Link
        </button>
    </div>
    
    <?php if ($error): ?>
        <div class="alert alert-danger alert-auto-dismiss"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <?php if ($success): ?>
        <div class="alert alert-success alert-auto-dismiss"><?php echo htmlspecialchars($success); ?></div>
    <?php endif; ?>
    
    <!-- Filter Form -->
    <?php if (!$selectedSubmenu && empty($type)): ?>
        <div class="card shadow mb-4">
            <div class="card-header py-3">
                <h6 class="m-0 font-weight-bold text-primary">Filter Links</h6>
            </div>
            <div class="card-body">
                <form method="get" action="" class="row g-3">
                    <div class="col-md-4">
                        <label for="submenu_id" class="form-label">Submenu</label>
                        <select class="form-select" id="submenu_id" name="submenu_id">
                            <option value="">All Submenus</option>
                            <?php foreach ($submenus as $submenu): ?>
                                <option value="<?php echo $submenu['id']; ?>">
                                    <?php echo htmlspecialchars($submenu['menu_title'] . ' > ' . $submenu['title']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label for="type" class="form-label">Link Type</label>
                        <select class="form-select" id="type" name="type">
                            <option value="">All Types</option>
                            <option value="website">Website</option>
                            <option value="download">Download</option>
                            <option value="whatsapp">WhatsApp</option>
                        </select>
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary">Filter</button>
                    </div>
                </form>
            </div>
        </div>
    <?php endif; ?>
    
    <!-- Quick Filter Buttons -->
    <?php if (!$selectedSubmenu): ?>
        <div class="mb-4">
            <a href="links.php" class="btn btn-outline-primary me-2 <?php echo empty($type) && empty($submenuId) ? 'active' : ''; ?>">
                All Links
            </a>
            <a href="links.php?type=website" class="btn btn-outline-primary me-2 <?php echo $type === 'website' ? 'active' : ''; ?>">
                <i class="fas fa-globe"></i> Website Links
            </a>
            <a href="links.php?type=download" class="btn btn-outline-primary me-2 <?php echo $type === 'download' ? 'active' : ''; ?>">
                <i class="fas fa-download"></i> Download Links
            </a>
            <a href="links.php?type=whatsapp" class="btn btn-outline-primary <?php echo $type === 'whatsapp' ? 'active' : ''; ?>">
                <i class="fab fa-whatsapp"></i> WhatsApp Links
            </a>
        </div>
    <?php endif; ?>
    
    <!-- Links Table -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">
                <?php 
                if ($selectedSubmenu) {
                    echo "Links for {$selectedSubmenu['title']}";
                } elseif (!empty($type)) {
                    echo ucfirst($type) . " Links";
                } else {
                    echo "All Links";
                }
                ?>
            </h6>
        </div>
        <div class="card-body">
            <?php if (empty($links)): ?>
                <p class="text-center text-muted">No links found.</p>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-bordered table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Menu > Submenu</th>
                                <th>Title</th>
                                <th>URL</th>
                                <th>Type</th>
                                <th>Order</th>
                                <th>Status</th>
                                <th>Created By</th>
                                <th>Created At</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($links as $link): ?>
                                <tr>
                                    <td><?php echo $link['id']; ?></td>
                                    <td><?php echo htmlspecialchars($link['menu_title'] . ' > ' . $link['submenu_title']); ?></td>
                                    <td><?php echo htmlspecialchars($link['title']); ?></td>
                                    <td>
                                        <a href="<?php echo htmlspecialchars($link['url']); ?>" target="_blank" class="text-truncate d-inline-block" style="max-width: 200px;">
                                            <?php echo htmlspecialchars($link['url']); ?>
                                        </a>
                                    </td>
                                    <td>
                                        <?php if ($link['type'] === 'website'): ?>
                                            <span class="badge bg-primary">Website</span>
                                        <?php elseif ($link['type'] === 'download'): ?>
                                            <span class="badge bg-success">Download</span>
                                        <?php elseif ($link['type'] === 'whatsapp'): ?>
                                            <span class="badge bg-success">WhatsApp</span>
                                        <?php else: ?>
                                            <span class="badge bg-secondary"><?php echo htmlspecialchars($link['type']); ?></span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo $link['display_order']; ?></td>
                                    <td>
                                        <?php if ($link['status'] === 'active'): ?>
                                            <span class="badge bg-success">Active</span>
                                        <?php else: ?>
                                            <span class="badge bg-secondary">Inactive</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($link['created_by_name'] ?? 'System'); ?></td>
                                    <td><?php echo date('M d, Y H:i', strtotime($link['created_at'])); ?></td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-primary edit-link" 
                                                data-id="<?php echo $link['id']; ?>"
                                                data-submenu-id="<?php echo $link['submenu_id']; ?>"
                                                data-title="<?php echo htmlspecialchars($link['title']); ?>"
                                                data-url="<?php echo htmlspecialchars($link['url']); ?>"
                                                data-type="<?php echo $link['type']; ?>"
                                                data-order="<?php echo $link['display_order']; ?>"
                                                data-status="<?php echo $link['status']; ?>"
                                                data-bs-toggle="modal" data-bs-target="#editLinkModal">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button type="button" class="btn btn-sm btn-danger delete-link"
                                                data-id="<?php echo $link['id']; ?>"
                                                data-title="<?php echo htmlspecialchars($link['title']); ?>"
                                                data-bs-toggle="modal" data-bs-target="#deleteLinkModal">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Add Link Modal -->
<div class="modal fade" id="addLinkModal" tabindex="-1" aria-labelledby="addLinkModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addLinkModalLabel">Add New Link</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="post" action="">
                <div class="modal-body">
                    <input type="hidden" name="action" value="add">
                    
                    <div class="mb-3">
                        <label for="submenu_id" class="form-label">Submenu</label>
                        <select class="form-select" id="submenu_id" name="submenu_id" required>
                            <option value="">Select Submenu</option>
                            <?php foreach ($submenus as $submenu): ?>
                                <option value="<?php echo $submenu['id']; ?>" <?php echo ($submenuId == $submenu['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($submenu['menu_title'] . ' > ' . $submenu['title']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="title" class="form-label">Link Title</label>
                        <input type="text" class="form-control" id="title" name="title" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="url" class="form-label">URL</label>
                        <input type="url" class="form-control" id="url" name="url" required>
                        <small class="text-muted">For WhatsApp links, use format: https://wa.me/1234567890</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="type" class="form-label">Link Type</label>
                        <select class="form-select" id="type" name="type" required>
                            <option value="website">Website</option>
                            <option value="download">Download</option>
                            <option value="whatsapp">WhatsApp</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="order" class="form-label">Display Order</label>
                        <input type="number" class="form-control" id="order" name="order" value="0" min="0">
                        <small class="text-muted">Lower numbers appear first</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="status" class="form-label">Status</label>
                        <select class="form-select" id="status" name="status">
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Add Link</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Link Modal -->
<div class="modal fade" id="editLinkModal" tabindex="-1" aria-labelledby="editLinkModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editLinkModalLabel">Edit Link</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="post" action="">
                <div class="modal-body">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="id" id="edit-id">
                    
                    <div class="mb-3">
                        <label for="edit-submenu-id" class="form-label">Submenu</label>
                        <select class="form-select" id="edit-submenu-id" name="submenu_id" required>
                            <option value="">Select Submenu</option>
                            <?php foreach ($submenus as $submenu): ?>
                                <option value="<?php echo $submenu['id']; ?>">
                                    <?php echo htmlspecialchars($submenu['menu_title'] . ' > ' . $submenu['title']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-title" class="form-label">Link Title</label>
                        <input type="text" class="form-control" id="edit-title" name="title" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-url" class="form-label">URL</label>
                        <input type="url" class="form-control" id="edit-url" name="url" required>
                        <small class="text-muted">For WhatsApp links, use format: https://wa.me/1234567890</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-type" class="form-label">Link Type</label>
                        <select class="form-select" id="edit-type" name="type" required>
                            <option value="website">Website</option>
                            <option value="download">Download</option>
                            <option value="whatsapp">WhatsApp</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-order" class="form-label">Display Order</label>
                        <input type="number" class="form-control" id="edit-order" name="order" value="0" min="0">
                        <small class="text-muted">Lower numbers appear first</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-status" class="form-label">Status</label>
                        <select class="form-select" id="edit-status" name="status">
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Update Link</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Link Modal -->
<div class="modal fade" id="deleteLinkModal" tabindex="-1" aria-labelledby="deleteLinkModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteLinkModalLabel">Delete Link</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete the link <strong id="delete-link-title"></strong>?</p>
                <p class="text-danger">This action cannot be undone.</p>
            </div>
            <form method="post" action="">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="id" id="delete-id">
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">Delete Link</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php
// Add page-specific scripts
$pageScripts = ['../assets/js/links.js'];

// Include footer
include 'includes/footer.php';
?>