<?php
/**
 * Menu Management
 */

// Include authentication
require_once 'auth.php';

// Initialize variables
$error = '';
$success = '';
$menus = [];

// Process form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Check action type
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add') {
        // Add new menu
        $title = trim($_POST['title'] ?? '');
        $order = (int)($_POST['order'] ?? 0);
        $status = $_POST['status'] ?? 'active';
        
        if (empty($title)) {
            $error = 'Menu title is required.';
        } else {
            try {
                $stmt = $db->prepare("INSERT INTO menus (title, display_order, status, created_by) VALUES (?, ?, ?, ?)");
                $stmt->execute([$title, $order, $status, $_SESSION['admin_id']]);
                
                $success = 'Menu added successfully.';
                logActivity('admin', null, $_SESSION['admin_id'], 'menu_add', "Added new menu: $title", $_SERVER['REMOTE_ADDR']);
            } catch (Exception $e) {
                $error = 'Error adding menu. Please try again.';
                logError("Add menu error: " . $e->getMessage());
            }
        }
    } elseif ($action === 'edit') {
        // Edit existing menu
        $id = (int)($_POST['id'] ?? 0);
        $title = trim($_POST['title'] ?? '');
        $order = (int)($_POST['order'] ?? 0);
        $status = $_POST['status'] ?? 'active';
        
        if (empty($title) || $id <= 0) {
            $error = 'Invalid menu data.';
        } else {
            try {
                $stmt = $db->prepare("UPDATE menus SET title = ?, display_order = ?, status = ?, updated_at = NOW() WHERE id = ?");
                $stmt->execute([$title, $order, $status, $id]);
                
                $success = 'Menu updated successfully.';
                logActivity('admin', null, $_SESSION['admin_id'], 'menu_edit', "Updated menu ID: $id", $_SERVER['REMOTE_ADDR']);
            } catch (Exception $e) {
                $error = 'Error updating menu. Please try again.';
                logError("Edit menu error: " . $e->getMessage());
            }
        }
    } elseif ($action === 'delete') {
        // Delete menu
        $id = (int)($_POST['id'] ?? 0);
        
        if ($id <= 0) {
            $error = 'Invalid menu ID.';
        } else {
            try {
                // Check if menu has submenus
                $stmt = $db->prepare("SELECT COUNT(*) as count FROM submenus WHERE menu_id = ?");
                $stmt->execute([$id]);
                $submenuCount = $stmt->fetch()['count'];
                
                if ($submenuCount > 0) {
                    $error = 'Cannot delete menu with existing submenus. Please delete submenus first.';
                } else {
                    $stmt = $db->prepare("DELETE FROM menus WHERE id = ?");
                    $stmt->execute([$id]);
                    
                    $success = 'Menu deleted successfully.';
                    logActivity('admin', null, $_SESSION['admin_id'], 'menu_delete', "Deleted menu ID: $id", $_SERVER['REMOTE_ADDR']);
                }
            } catch (Exception $e) {
                $error = 'Error deleting menu. Please try again.';
                logError("Delete menu error: " . $e->getMessage());
            }
        }
    }
}

// Get all menus
try {
    $stmt = $db->query("SELECT m.*, 
                        (SELECT COUNT(*) FROM submenus s WHERE s.menu_id = m.id) as submenu_count,
                        a.username as created_by_name
                        FROM menus m 
                        LEFT JOIN admin_users a ON m.created_by = a.id 
                        ORDER BY m.display_order, m.title");
    $menus = $stmt->fetchAll();
} catch (Exception $e) {
    $error = 'Error loading menus. Please try again.';
    logError("Load menus error: " . $e->getMessage());
}

// Page title
$pageTitle = "Menu Management";

// Include header
include 'includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1 class="h3">Menu Management</h1>
        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addMenuModal">
            <i class="fas fa-plus"></i> Add New Menu
        </button>
    </div>
    
    <?php if ($error): ?>
        <div class="alert alert-danger alert-auto-dismiss"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <?php if ($success): ?>
        <div class="alert alert-success alert-auto-dismiss"><?php echo htmlspecialchars($success); ?></div>
    <?php endif; ?>
    
    <!-- Menus Table -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">All Menus</h6>
        </div>
        <div class="card-body">
            <?php if (empty($menus)): ?>
                <p class="text-center text-muted">No menus found. Create your first menu to get started.</p>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-bordered table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Title</th>
                                <th>Order</th>
                                <th>Submenus</th>
                                <th>Status</th>
                                <th>Created By</th>
                                <th>Created At</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($menus as $menu): ?>
                                <tr>
                                    <td><?php echo $menu['id']; ?></td>
                                    <td><?php echo htmlspecialchars($menu['title']); ?></td>
                                    <td><?php echo $menu['display_order']; ?></td>
                                    <td>
                                        <a href="submenus.php?menu_id=<?php echo $menu['id']; ?>" class="btn btn-sm btn-outline-info">
                                            <?php echo $menu['submenu_count']; ?> Submenus
                                        </a>
                                    </td>
                                    <td>
                                        <?php if ($menu['status'] === 'active'): ?>
                                            <span class="badge bg-success">Active</span>
                                        <?php else: ?>
                                            <span class="badge bg-secondary">Inactive</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($menu['created_by_name'] ?? 'System'); ?></td>
                                    <td><?php echo date('M d, Y H:i', strtotime($menu['created_at'])); ?></td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-primary edit-menu" 
                                                data-id="<?php echo $menu['id']; ?>"
                                                data-title="<?php echo htmlspecialchars($menu['title']); ?>"
                                                data-order="<?php echo $menu['display_order']; ?>"
                                                data-status="<?php echo $menu['status']; ?>"
                                                data-bs-toggle="modal" data-bs-target="#editMenuModal">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button type="button" class="btn btn-sm btn-danger delete-menu"
                                                data-id="<?php echo $menu['id']; ?>"
                                                data-title="<?php echo htmlspecialchars($menu['title']); ?>"
                                                data-bs-toggle="modal" data-bs-target="#deleteMenuModal">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Add Menu Modal -->
<div class="modal fade" id="addMenuModal" tabindex="-1" aria-labelledby="addMenuModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addMenuModalLabel">Add New Menu</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="post" action="">
                <div class="modal-body">
                    <input type="hidden" name="action" value="add">
                    
                    <div class="mb-3">
                        <label for="title" class="form-label">Menu Title</label>
                        <input type="text" class="form-control" id="title" name="title" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="order" class="form-label">Display Order</label>
                        <input type="number" class="form-control" id="order" name="order" value="0" min="0">
                        <small class="text-muted">Lower numbers appear first</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="status" class="form-label">Status</label>
                        <select class="form-select" id="status" name="status">
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Add Menu</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Menu Modal -->
<div class="modal fade" id="editMenuModal" tabindex="-1" aria-labelledby="editMenuModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editMenuModalLabel">Edit Menu</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="post" action="">
                <div class="modal-body">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="id" id="edit-id">
                    
                    <div class="mb-3">
                        <label for="edit-title" class="form-label">Menu Title</label>
                        <input type="text" class="form-control" id="edit-title" name="title" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-order" class="form-label">Display Order</label>
                        <input type="number" class="form-control" id="edit-order" name="order" value="0" min="0">
                        <small class="text-muted">Lower numbers appear first</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-status" class="form-label">Status</label>
                        <select class="form-select" id="edit-status" name="status">
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Update Menu</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Menu Modal -->
<div class="modal fade" id="deleteMenuModal" tabindex="-1" aria-labelledby="deleteMenuModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteMenuModalLabel">Delete Menu</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete the menu <strong id="delete-menu-title"></strong>?</p>
                <p class="text-danger">This action cannot be undone. All associated data will be permanently deleted.</p>
            </div>
            <form method="post" action="">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="id" id="delete-id">
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">Delete Menu</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php
// Add page-specific scripts
$pageScripts = ['../assets/js/menus.js'];

// Include footer
include 'includes/footer.php';
?>