<?php
/**
 * Bot Settings
 */

// Include authentication
require_once 'auth.php';

// Initialize variables
$error = '';
$success = '';
$settings = [];

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Check if action is update_settings
    if (isset($_POST['action']) && $_POST['action'] === 'update_settings') {
        // Get settings from form
        $newSettings = [
            'welcome_message' => isset($_POST['welcome_message']) ? trim($_POST['welcome_message']) : '',
            'help_message' => isset($_POST['help_message']) ? trim($_POST['help_message']) : '',
            'about_message' => isset($_POST['about_message']) ? trim($_POST['about_message']) : '',
            'error_message' => isset($_POST['error_message']) ? trim($_POST['error_message']) : '',
            'maintenance_mode' => isset($_POST['maintenance_mode']) ? 1 : 0,
            'maintenance_message' => isset($_POST['maintenance_message']) ? trim($_POST['maintenance_message']) : '',
        ];
        
        // Update settings in database
        try {
            foreach ($newSettings as $key => $value) {
                $stmt = $db->prepare("INSERT INTO settings (setting_key, setting_value) 
                                    VALUES (?, ?) 
                                    ON DUPLICATE KEY UPDATE setting_value = ?");
                $stmt->execute([$key, $value, $value]);
            }
            
            // Log admin activity
            logActivity('admin', 'update_settings', "Updated bot settings", $_SESSION['admin_id']);
            
            $success = 'Settings updated successfully.';
        } catch (Exception $e) {
            logError("Settings update error: " . $e->getMessage());
            $error = 'Error updating settings. Please try again.';
        }
    }
    
    // Check if action is set_webhook
    if (isset($_POST['action']) && $_POST['action'] === 'set_webhook') {
        $webhookUrl = isset($_POST['webhook_url']) ? trim($_POST['webhook_url']) : '';
        
        if (empty($webhookUrl)) {
            $error = 'Webhook URL cannot be empty.';
        } else {
            // Initialize TelegramBot class
            require_once '../classes/TelegramBot.php';
            $bot = new TelegramBot(BOT_TOKEN);
            
            // Set webhook
            $result = $bot->setWebhook($webhookUrl);
            
            if ($result) {
                // Update webhook URL in settings
                try {
                    $stmt = $db->prepare("INSERT INTO settings (setting_key, setting_value) 
                                        VALUES ('webhook_url', ?) 
                                        ON DUPLICATE KEY UPDATE setting_value = ?");
                    $stmt->execute([$webhookUrl, $webhookUrl]);
                    
                    // Log admin activity
                    logActivity('admin', 'set_webhook', "Set webhook URL to {$webhookUrl}", $_SESSION['admin_id']);
                    
                    $success = 'Webhook set successfully.';
                } catch (Exception $e) {
                    logError("Webhook URL update error: " . $e->getMessage());
                    $error = 'Error updating webhook URL in database. Please try again.';
                }
            } else {
                $error = 'Error setting webhook. Please check the URL and try again.';
            }
        }
    }
    
    // Check if action is delete_webhook
    if (isset($_POST['action']) && $_POST['action'] === 'delete_webhook') {
        // Initialize TelegramBot class
        require_once '../classes/TelegramBot.php';
        $bot = new TelegramBot(BOT_TOKEN);
        
        // Delete webhook
        $result = $bot->deleteWebhook();
        
        if ($result) {
            // Update webhook URL in settings
            try {
                $stmt = $db->prepare("INSERT INTO settings (setting_key, setting_value) 
                                    VALUES ('webhook_url', '') 
                                    ON DUPLICATE KEY UPDATE setting_value = ''");
                $stmt->execute();
                
                // Log admin activity
                logActivity('admin', 'delete_webhook', "Deleted webhook", $_SESSION['admin_id']);
                
                $success = 'Webhook deleted successfully.';
            } catch (Exception $e) {
                logError("Webhook URL delete error: " . $e->getMessage());
                $error = 'Error updating webhook URL in database. Please try again.';
            }
        } else {
            $error = 'Error deleting webhook. Please try again.';
        }
    }
    
    // Check if action is get_bot_info
    if (isset($_POST['action']) && $_POST['action'] === 'get_bot_info') {
        // Initialize TelegramBot class
        require_once '../classes/TelegramBot.php';
        $bot = new TelegramBot(BOT_TOKEN);
        
        // Get bot info
        $botInfo = $bot->getBotInfo();
        
        if ($botInfo) {
            // Store bot info in session for display
            $_SESSION['bot_info'] = $botInfo;
            
            // Log admin activity
            logActivity('admin', 'get_bot_info', "Retrieved bot information", $_SESSION['admin_id']);
            
            $success = 'Bot information retrieved successfully.';
        } else {
            $error = 'Error retrieving bot information. Please try again.';
        }
    }
}

// Get settings from database
try {
    $stmt = $db->query("SELECT * FROM settings");
    $settingsRows = $stmt->fetchAll();
    
    // Convert to associative array
    foreach ($settingsRows as $row) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
} catch (Exception $e) {
    logError("Settings load error: " . $e->getMessage());
    $error = 'Error loading settings. Please try again.';
}

// Page title
$pageTitle = "Bot Settings";

// Include header
include 'includes/header.php';
?>

<div class="container-fluid py-4">
    <h1 class="h3 mb-4"><?php echo $pageTitle; ?></h1>
    
    <?php if ($error): ?>
        <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <?php if ($success): ?>
        <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
    <?php endif; ?>
    
    <div class="row">
        <div class="col-lg-8">
            <!-- Bot Settings Form -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Bot Settings</h6>
                </div>
                <div class="card-body">
                    <form method="post" action="">
                        <input type="hidden" name="action" value="update_settings">
                        
                        <!-- Welcome Message -->
                        <div class="mb-3">
                            <label for="welcome_message" class="form-label">Welcome Message</label>
                            <textarea class="form-control" id="welcome_message" name="welcome_message" rows="3"><?php echo htmlspecialchars($settings['welcome_message'] ?? ''); ?></textarea>
                            <small class="text-muted">This message is sent when a user starts the bot for the first time.</small>
                        </div>
                        
                        <!-- Help Message -->
                        <div class="mb-3">
                            <label for="help_message" class="form-label">Help Message</label>
                            <textarea class="form-control" id="help_message" name="help_message" rows="3"><?php echo htmlspecialchars($settings['help_message'] ?? ''); ?></textarea>
                            <small class="text-muted">This message is sent when a user sends the /help command.</small>
                        </div>
                        
                        <!-- About Message -->
                        <div class="mb-3">
                            <label for="about_message" class="form-label">About Message</label>
                            <textarea class="form-control" id="about_message" name="about_message" rows="3"><?php echo htmlspecialchars($settings['about_message'] ?? ''); ?></textarea>
                            <small class="text-muted">This message is sent when a user sends the /about command.</small>
                        </div>
                        
                        <!-- Error Message -->
                        <div class="mb-3">
                            <label for="error_message" class="form-label">Error Message</label>
                            <textarea class="form-control" id="error_message" name="error_message" rows="2"><?php echo htmlspecialchars($settings['error_message'] ?? ''); ?></textarea>
                            <small class="text-muted">This message is sent when an error occurs.</small>
                        </div>
                        
                        <!-- Maintenance Mode -->
                        <div class="mb-3 form-check">
                            <input type="checkbox" class="form-check-input" id="maintenance_mode" name="maintenance_mode" value="1" <?php echo (isset($settings['maintenance_mode']) && $settings['maintenance_mode'] == 1) ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="maintenance_mode">Maintenance Mode</label>
                            <small class="d-block text-muted">When enabled, only admins can use the bot.</small>
                        </div>
                        
                        <!-- Maintenance Message -->
                        <div class="mb-3">
                            <label for="maintenance_message" class="form-label">Maintenance Message</label>
                            <textarea class="form-control" id="maintenance_message" name="maintenance_message" rows="2"><?php echo htmlspecialchars($settings['maintenance_message'] ?? ''); ?></textarea>
                            <small class="text-muted">This message is sent when maintenance mode is enabled.</small>
                        </div>
                        
                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i> Save Settings
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <div class="col-lg-4">
            <!-- Webhook Settings -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Webhook Settings</h6>
                </div>
                <div class="card-body">
                    <form method="post" action="" id="webhookForm">
                        <input type="hidden" name="action" value="set_webhook">
                        
                        <div class="mb-3">
                            <label for="webhook_url" class="form-label">Webhook URL</label>
                            <input type="url" class="form-control" id="webhook_url" name="webhook_url" value="<?php echo htmlspecialchars($settings['webhook_url'] ?? ''); ?>" required>
                            <small class="text-muted">The URL where Telegram will send updates to your bot.</small>
                        </div>
                        
                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-primary mb-2">
                                <i class="fas fa-link"></i> Set Webhook
                            </button>
                            
                            <button type="button" class="btn btn-danger" id="deleteWebhookBtn">
                                <i class="fas fa-unlink"></i> Delete Webhook
                            </button>
                        </div>
                    </form>
                    
                    <!-- Delete Webhook Form (hidden) -->
                    <form method="post" action="" id="deleteWebhookForm" style="display: none;">
                        <input type="hidden" name="action" value="delete_webhook">
                    </form>
                </div>
            </div>
            
            <!-- Bot Information -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Bot Information</h6>
                </div>
                <div class="card-body">
                    <form method="post" action="">
                        <input type="hidden" name="action" value="get_bot_info">
                        
                        <div class="d-grid gap-2 mb-3">
                            <button type="submit" class="btn btn-info">
                                <i class="fas fa-info-circle"></i> Get Bot Info
                            </button>
                        </div>
                    </form>
                    
                    <?php if (isset($_SESSION['bot_info']) && !empty($_SESSION['bot_info'])): ?>
                        <div class="table-responsive">
                            <table class="table table-bordered">
                                <tbody>
                                    <?php foreach ($_SESSION['bot_info'] as $key => $value): ?>
                                        <tr>
                                            <th><?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $key))); ?></th>
                                            <td>
                                                <?php 
                                                if (is_bool($value)) {
                                                    echo $value ? 'Yes' : 'No';
                                                } elseif (is_array($value) || is_object($value)) {
                                                    echo '<pre>' . htmlspecialchars(json_encode($value, JSON_PRETTY_PRINT)) . '</pre>';
                                                } else {
                                                    echo htmlspecialchars($value);
                                                }
                                                ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<?php
// Add page-specific scripts
$pageScripts = ['../assets/js/settings.js'];

// Include footer
include 'includes/footer.php';
?>