<?php
/**
 * Submenu Management
 */

// Include authentication
require_once 'auth.php';

// Initialize variables
$error = '';
$success = '';
$submenus = [];
$menus = [];
$selectedMenu = null;

// Get filter parameters
$menuId = isset($_GET['menu_id']) ? (int)$_GET['menu_id'] : 0;

// Process form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Check action type
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add') {
        // Add new submenu
        $menuId = (int)($_POST['menu_id'] ?? 0);
        $title = trim($_POST['title'] ?? '');
        $order = (int)($_POST['order'] ?? 0);
        $status = $_POST['status'] ?? 'active';
        $contentType = $_POST['content_type'] ?? 'text';
        
        if (empty($title) || $menuId <= 0) {
            $error = 'Menu and submenu title are required.';
        } else {
            try {
                $stmt = $db->prepare("INSERT INTO submenus 
                                    (menu_id, title, display_order, status, content_type, created_by) 
                                    VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->execute([$menuId, $title, $order, $status, $contentType, $_SESSION['admin_id']]);
                
                $success = 'Submenu added successfully.';
                logActivity('admin', null, $_SESSION['admin_id'], 'submenu_add', "Added new submenu: $title", $_SERVER['REMOTE_ADDR']);
            } catch (Exception $e) {
                $error = 'Error adding submenu. Please try again.';
                logError("Add submenu error: " . $e->getMessage());
            }
        }
    } elseif ($action === 'edit') {
        // Edit existing submenu
        $id = (int)($_POST['id'] ?? 0);
        $menuId = (int)($_POST['menu_id'] ?? 0);
        $title = trim($_POST['title'] ?? '');
        $order = (int)($_POST['order'] ?? 0);
        $status = $_POST['status'] ?? 'active';
        $contentType = $_POST['content_type'] ?? 'text';
        
        if (empty($title) || $id <= 0 || $menuId <= 0) {
            $error = 'Invalid submenu data.';
        } else {
            try {
                $stmt = $db->prepare("UPDATE submenus 
                                    SET menu_id = ?, title = ?, display_order = ?, 
                                        status = ?, content_type = ?, updated_at = NOW() 
                                    WHERE id = ?");
                $stmt->execute([$menuId, $title, $order, $status, $contentType, $id]);
                
                $success = 'Submenu updated successfully.';
                logActivity('admin', null, $_SESSION['admin_id'], 'submenu_edit', "Updated submenu ID: $id", $_SERVER['REMOTE_ADDR']);
            } catch (Exception $e) {
                $error = 'Error updating submenu. Please try again.';
                logError("Edit submenu error: " . $e->getMessage());
            }
        }
    } elseif ($action === 'delete') {
        // Delete submenu
        $id = (int)($_POST['id'] ?? 0);
        
        if ($id <= 0) {
            $error = 'Invalid submenu ID.';
        } else {
            try {
                // Check if submenu has content
                $stmt = $db->prepare("SELECT COUNT(*) as count FROM content WHERE submenu_id = ?");
                $stmt->execute([$id]);
                $contentCount = $stmt->fetch()['count'];
                
                if ($contentCount > 0) {
                    // Delete associated content first
                    $stmt = $db->prepare("DELETE FROM content WHERE submenu_id = ?");
                    $stmt->execute([$id]);
                }
                
                // Delete submenu
                $stmt = $db->prepare("DELETE FROM submenus WHERE id = ?");
                $stmt->execute([$id]);
                
                $success = 'Submenu deleted successfully.';
                logActivity('admin', null, $_SESSION['admin_id'], 'submenu_delete', "Deleted submenu ID: $id", $_SERVER['REMOTE_ADDR']);
            } catch (Exception $e) {
                $error = 'Error deleting submenu. Please try again.';
                logError("Delete submenu error: " . $e->getMessage());
            }
        }
    }
}

// Get all menus for dropdown
try {
    $stmt = $db->query("SELECT id, title FROM menus WHERE status = 'active' ORDER BY display_order, title");
    $menus = $stmt->fetchAll();
    
    // Get selected menu details if menu_id is provided
    if ($menuId > 0) {
        $stmt = $db->prepare("SELECT id, title FROM menus WHERE id = ?");
        $stmt->execute([$menuId]);
        $selectedMenu = $stmt->fetch();
    }
} catch (Exception $e) {
    $error = 'Error loading menus. Please try again.';
    logError("Load menus error: " . $e->getMessage());
}

// Get submenus based on filter
try {
    $query = "SELECT s.*, 
              m.title as menu_title,
              (SELECT COUNT(*) FROM content c WHERE c.submenu_id = s.id) as content_count,
              a.username as created_by_name
              FROM submenus s 
              LEFT JOIN menus m ON s.menu_id = m.id 
              LEFT JOIN admin_users a ON s.created_by = a.id ";
    
    $params = [];
    
    if ($menuId > 0) {
        $query .= "WHERE s.menu_id = ? ";
        $params[] = $menuId;
    }
    
    $query .= "ORDER BY s.menu_id, s.display_order, s.title";
    
    if (!empty($params)) {
        $stmt = $db->prepare($query);
        $stmt->execute($params);
    } else {
        $stmt = $db->query($query);
    }
    
    $submenus = $stmt->fetchAll();
} catch (Exception $e) {
    $error = 'Error loading submenus. Please try again.';
    logError("Load submenus error: " . $e->getMessage());
}

// Page title
$pageTitle = $selectedMenu ? "Submenus for {$selectedMenu['title']}" : "Submenu Management";

// Include header
include 'includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3"><?php echo htmlspecialchars($pageTitle); ?></h1>
            <?php if ($selectedMenu): ?>
                <a href="submenus.php" class="btn btn-sm btn-outline-secondary">
                    <i class="fas fa-arrow-left"></i> Show All Submenus
                </a>
            <?php endif; ?>
        </div>
        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addSubmenuModal">
            <i class="fas fa-plus"></i> Add New Submenu
        </button>
    </div>
    
    <?php if ($error): ?>
        <div class="alert alert-danger alert-auto-dismiss"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <?php if ($success): ?>
        <div class="alert alert-success alert-auto-dismiss"><?php echo htmlspecialchars($success); ?></div>
    <?php endif; ?>
    
    <!-- Filter Form -->
    <?php if (!$selectedMenu): ?>
        <div class="card shadow mb-4">
            <div class="card-header py-3">
                <h6 class="m-0 font-weight-bold text-primary">Filter Submenus</h6>
            </div>
            <div class="card-body">
                <form method="get" action="" class="row g-3">
                    <div class="col-md-4">
                        <label for="menu_id" class="form-label">Menu</label>
                        <select class="form-select" id="menu_id" name="menu_id">
                            <option value="">All Menus</option>
                            <?php foreach ($menus as $menu): ?>
                                <option value="<?php echo $menu['id']; ?>">
                                    <?php echo htmlspecialchars($menu['title']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary">Filter</button>
                    </div>
                </form>
            </div>
        </div>
    <?php endif; ?>
    
    <!-- Submenus Table -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">
                <?php echo $selectedMenu ? "Submenus for {$selectedMenu['title']}" : "All Submenus"; ?>
            </h6>
        </div>
        <div class="card-body">
            <?php if (empty($submenus)): ?>
                <p class="text-center text-muted">No submenus found.</p>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-bordered table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Menu</th>
                                <th>Title</th>
                                <th>Order</th>
                                <th>Content Type</th>
                                <th>Content</th>
                                <th>Status</th>
                                <th>Created By</th>
                                <th>Created At</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($submenus as $submenu): ?>
                                <tr>
                                    <td><?php echo $submenu['id']; ?></td>
                                    <td><?php echo htmlspecialchars($submenu['menu_title']); ?></td>
                                    <td><?php echo htmlspecialchars($submenu['title']); ?></td>
                                    <td><?php echo $submenu['display_order']; ?></td>
                                    <td>
                                        <?php if ($submenu['content_type'] === 'text'): ?>
                                            <span class="badge bg-info">Text</span>
                                        <?php elseif ($submenu['content_type'] === 'link'): ?>
                                            <span class="badge bg-primary">Link</span>
                                        <?php elseif ($submenu['content_type'] === 'media'): ?>
                                            <span class="badge bg-warning">Media</span>
                                        <?php else: ?>
                                            <span class="badge bg-secondary"><?php echo ucfirst($submenu['content_type']); ?></span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <a href="content.php?submenu_id=<?php echo $submenu['id']; ?>" class="btn btn-sm btn-outline-info">
                                            <?php echo $submenu['content_count']; ?> Items
                                        </a>
                                    </td>
                                    <td>
                                        <?php if ($submenu['status'] === 'active'): ?>
                                            <span class="badge bg-success">Active</span>
                                        <?php else: ?>
                                            <span class="badge bg-secondary">Inactive</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($submenu['created_by_name'] ?? 'System'); ?></td>
                                    <td><?php echo date('M d, Y H:i', strtotime($submenu['created_at'])); ?></td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-primary edit-submenu" 
                                                data-id="<?php echo $submenu['id']; ?>"
                                                data-menu-id="<?php echo $submenu['menu_id']; ?>"
                                                data-title="<?php echo htmlspecialchars($submenu['title']); ?>"
                                                data-order="<?php echo $submenu['display_order']; ?>"
                                                data-status="<?php echo $submenu['status']; ?>"
                                                data-content-type="<?php echo $submenu['content_type']; ?>"
                                                data-bs-toggle="modal" data-bs-target="#editSubmenuModal">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button type="button" class="btn btn-sm btn-danger delete-submenu"
                                                data-id="<?php echo $submenu['id']; ?>"
                                                data-title="<?php echo htmlspecialchars($submenu['title']); ?>"
                                                data-bs-toggle="modal" data-bs-target="#deleteSubmenuModal">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Add Submenu Modal -->
<div class="modal fade" id="addSubmenuModal" tabindex="-1" aria-labelledby="addSubmenuModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addSubmenuModalLabel">Add New Submenu</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="post" action="">
                <div class="modal-body">
                    <input type="hidden" name="action" value="add">
                    
                    <div class="mb-3">
                        <label for="menu_id" class="form-label">Parent Menu</label>
                        <select class="form-select" id="menu_id" name="menu_id" required>
                            <option value="">Select Menu</option>
                            <?php foreach ($menus as $menu): ?>
                                <option value="<?php echo $menu['id']; ?>" <?php echo ($menuId == $menu['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($menu['title']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="title" class="form-label">Submenu Title</label>
                        <input type="text" class="form-control" id="title" name="title" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="order" class="form-label">Display Order</label>
                        <input type="number" class="form-control" id="order" name="order" value="0" min="0">
                        <small class="text-muted">Lower numbers appear first</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="content_type" class="form-label">Content Type</label>
                        <select class="form-select" id="content_type" name="content_type">
                            <option value="text">Text</option>
                            <option value="link">Link</option>
                            <option value="media">Media</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="status" class="form-label">Status</label>
                        <select class="form-select" id="status" name="status">
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Add Submenu</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Submenu Modal -->
<div class="modal fade" id="editSubmenuModal" tabindex="-1" aria-labelledby="editSubmenuModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editSubmenuModalLabel">Edit Submenu</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="post" action="">
                <div class="modal-body">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="id" id="edit-id">
                    
                    <div class="mb-3">
                        <label for="edit-menu-id" class="form-label">Parent Menu</label>
                        <select class="form-select" id="edit-menu-id" name="menu_id" required>
                            <option value="">Select Menu</option>
                            <?php foreach ($menus as $menu): ?>
                                <option value="<?php echo $menu['id']; ?>">
                                    <?php echo htmlspecialchars($menu['title']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-title" class="form-label">Submenu Title</label>
                        <input type="text" class="form-control" id="edit-title" name="title" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-order" class="form-label">Display Order</label>
                        <input type="number" class="form-control" id="edit-order" name="order" value="0" min="0">
                        <small class="text-muted">Lower numbers appear first</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-content-type" class="form-label">Content Type</label>
                        <select class="form-select" id="edit-content-type" name="content_type">
                            <option value="text">Text</option>
                            <option value="link">Link</option>
                            <option value="media">Media</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit-status" class="form-label">Status</label>
                        <select class="form-select" id="edit-status" name="status">
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Update Submenu</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Submenu Modal -->
<div class="modal fade" id="deleteSubmenuModal" tabindex="-1" aria-labelledby="deleteSubmenuModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteSubmenuModalLabel">Delete Submenu</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete the submenu <strong id="delete-submenu-title"></strong>?</p>
                <p class="text-danger">This action cannot be undone. All associated content will be permanently deleted.</p>
            </div>
            <form method="post" action="">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="id" id="delete-id">
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">Delete Submenu</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php
// Add page-specific scripts
$pageScripts = ['../assets/js/submenus.js'];

// Include footer
include 'includes/footer.php';
?>