<?php
/**
 * User Activity
 */

// Include authentication
require_once 'auth.php';

// Initialize variables
$error = '';
$user = null;
$activities = [];

// Get user ID from query string
$userId = isset($_GET['user_id']) ? (int)$_GET['user_id'] : 0;

if ($userId <= 0) {
    $error = 'Invalid user ID.';
} else {
    // Get user details
    try {
        $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
        
        if (!$user) {
            $error = 'User not found.';
        }
    } catch (Exception $e) {
        $error = 'Error loading user details. Please try again.';
        logError("Load user error: " . $e->getMessage());
    }
    
    // Get user activities
    if ($user) {
        try {
            $stmt = $db->prepare("SELECT * FROM activity_logs 
                                WHERE user_id = ? 
                                ORDER BY created_at DESC 
                                LIMIT 100");
            $stmt->execute([$userId]);
            $activities = $stmt->fetchAll();
        } catch (Exception $e) {
            $error = 'Error loading user activities. Please try again.';
            logError("Load activities error: " . $e->getMessage());
        }
    }
}

// Page title
$pageTitle = $user ? "Activity for " . $user['first_name'] . ' ' . $user['last_name'] : "User Activity";

// Include header
include 'includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3"><?php echo htmlspecialchars($pageTitle); ?></h1>
            <a href="users.php" class="btn btn-sm btn-outline-secondary">
                <i class="fas fa-arrow-left"></i> Back to Users
            </a>
        </div>
        <?php if ($user): ?>
            <div>
                <button type="button" class="btn btn-primary message-user"
                        data-id="<?php echo $user['id']; ?>"
                        data-name="<?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?>"
                        data-bs-toggle="modal" data-bs-target="#messageUserModal">
                    <i class="fas fa-envelope"></i> Send Message
                </button>
                
                <?php if ($user['status'] === 'active'): ?>
                    <button type="button" class="btn btn-warning block-user"
                            data-id="<?php echo $user['id']; ?>"
                            data-name="<?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?>"
                            data-bs-toggle="modal" data-bs-target="#blockUserModal">
                        <i class="fas fa-ban"></i> Block User
                    </button>
                <?php else: ?>
                    <button type="button" class="btn btn-success unblock-user"
                            data-id="<?php echo $user['id']; ?>"
                            data-name="<?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?>"
                            data-bs-toggle="modal" data-bs-target="#unblockUserModal">
                        <i class="fas fa-check"></i> Unblock User
                    </button>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
    
    <?php if ($error): ?>
        <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <?php if ($user): ?>
        <!-- User Details Card -->
        <div class="card shadow mb-4">
            <div class="card-header py-3">
                <h6 class="m-0 font-weight-bold text-primary">User Details</h6>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <table class="table table-borderless">
                            <tr>
                                <th>ID:</th>
                                <td><?php echo $user['id']; ?></td>
                            </tr>
                            <tr>
                                <th>Telegram ID:</th>
                                <td><?php echo $user['telegram_id']; ?></td>
                            </tr>
                            <tr>
                                <th>Name:</th>
                                <td><?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?></td>
                            </tr>
                            <tr>
                                <th>Username:</th>
                                <td>
                                    <?php if (!empty($user['username'])): ?>
                                        @<?php echo htmlspecialchars($user['username']); ?>
                                    <?php else: ?>
                                        <span class="text-muted">No username</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        </table>
                    </div>
                    <div class="col-md-6">
                        <table class="table table-borderless">
                            <tr>
                                <th>Status:</th>
                                <td>
                                    <?php if ($user['status'] === 'active'): ?>
                                        <span class="badge bg-success">Active</span>
                                    <?php elseif ($user['status'] === 'blocked'): ?>
                                        <span class="badge bg-danger">Blocked</span>
                                    <?php else: ?>
                                        <span class="badge bg-secondary"><?php echo htmlspecialchars($user['status']); ?></span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <tr>
                                <th>Language:</th>
                                <td><?php echo htmlspecialchars($user['language_code'] ?? 'Not set'); ?></td>
                            </tr>
                            <tr>
                                <th>Joined:</th>
                                <td><?php echo date('M d, Y H:i', strtotime($user['created_at'])); ?></td>
                            </tr>
                            <tr>
                                <th>Last Updated:</th>
                                <td>
                                    <?php if ($user['updated_at']): ?>
                                        <?php echo date('M d, Y H:i', strtotime($user['updated_at'])); ?>
                                    <?php else: ?>
                                        <span class="text-muted">Never</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        </table>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Activity Log -->
        <div class="card shadow mb-4">
            <div class="card-header py-3">
                <h6 class="m-0 font-weight-bold text-primary">Activity Log</h6>
            </div>
            <div class="card-body">
                <?php if (empty($activities)): ?>
                    <p class="text-center text-muted">No activity found for this user.</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-bordered table-hover">
                            <thead>
                                <tr>
                                    <th>Date/Time</th>
                                    <th>Type</th>
                                    <th>Action</th>
                                    <th>Details</th>
                                    <th>IP Address</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($activities as $activity): ?>
                                    <tr>
                                        <td><?php echo date('M d, Y H:i:s', strtotime($activity['created_at'])); ?></td>
                                        <td><?php echo htmlspecialchars($activity['type']); ?></td>
                                        <td><?php echo htmlspecialchars($activity['action']); ?></td>
                                        <td><?php echo htmlspecialchars($activity['details']); ?></td>
                                        <td><?php echo htmlspecialchars($activity['ip_address']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    <?php endif; ?>
</div>

<!-- Message User Modal -->
<div class="modal fade" id="messageUserModal" tabindex="-1" aria-labelledby="messageUserModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="messageUserModalLabel">Send Message</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="post" action="users.php">
                <div class="modal-body">
                    <input type="hidden" name="action" value="message">
                    <input type="hidden" name="user_id" id="message-user-id">
                    
                    <div class="mb-3">
                        <label for="message-recipient" class="form-label">Recipient</label>
                        <input type="text" class="form-control" id="message-recipient" readonly>
                    </div>
                    
                    <div class="mb-3">
                        <label for="message" class="form-label">Message</label>
                        <textarea class="form-control" id="message" name="message" rows="5" required></textarea>
                        <small class="text-muted">This message will be sent directly to the user via the bot.</small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Send Message</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Block User Modal -->
<div class="modal fade" id="blockUserModal" tabindex="-1" aria-labelledby="blockUserModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="blockUserModalLabel">Block User</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to block <strong id="block-user-name"></strong>?</p>
                <p>Blocked users will not be able to use the bot.</p>
            </div>
            <form method="post" action="users.php">
                <input type="hidden" name="action" value="block">
                <input type="hidden" name="user_id" id="block-user-id">
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-warning">Block User</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Unblock User Modal -->
<div class="modal fade" id="unblockUserModal" tabindex="-1" aria-labelledby="unblockUserModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="unblockUserModalLabel">Unblock User</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to unblock <strong id="unblock-user-name"></strong>?</p>
                <p>This will allow the user to use the bot again.</p>
            </div>
            <form method="post" action="users.php">
                <input type="hidden" name="action" value="unblock">
                <input type="hidden" name="user_id" id="unblock-user-id">
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-success">Unblock User</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php
// Add page-specific scripts
$pageScripts = ['../assets/js/users.js'];

// Include footer
include 'includes/footer.php';
?>