<?php
/**
 * User Management
 */

// Include authentication
require_once 'auth.php';

// Initialize variables
$error = '';
$success = '';
$users = [];

// Get filter parameters
$status = isset($_GET['status']) ? $_GET['status'] : '';
$search = isset($_GET['search']) ? trim($_GET['search']) : '';

// Process form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Check action type
    $action = $_POST['action'] ?? '';
    
    if ($action === 'block') {
        // Block user
        $userId = (int)($_POST['user_id'] ?? 0);
        
        if ($userId <= 0) {
            $error = 'Invalid user ID.';
        } else {
            try {
                $stmt = $db->prepare("UPDATE users SET status = 'blocked', updated_at = NOW() WHERE id = ?");
                $stmt->execute([$userId]);
                
                $success = 'User blocked successfully.';
                logActivity('admin', null, $_SESSION['admin_id'], 'user_block', "Blocked user ID: $userId", $_SERVER['REMOTE_ADDR']);
            } catch (Exception $e) {
                $error = 'Error blocking user. Please try again.';
                logError("Block user error: " . $e->getMessage());
            }
        }
    } elseif ($action === 'unblock') {
        // Unblock user
        $userId = (int)($_POST['user_id'] ?? 0);
        
        if ($userId <= 0) {
            $error = 'Invalid user ID.';
        } else {
            try {
                $stmt = $db->prepare("UPDATE users SET status = 'active', updated_at = NOW() WHERE id = ?");
                $stmt->execute([$userId]);
                
                $success = 'User unblocked successfully.';
                logActivity('admin', null, $_SESSION['admin_id'], 'user_unblock', "Unblocked user ID: $userId", $_SERVER['REMOTE_ADDR']);
            } catch (Exception $e) {
                $error = 'Error unblocking user. Please try again.';
                logError("Unblock user error: " . $e->getMessage());
            }
        }
    } elseif ($action === 'message') {
        // Send message to user
        $userId = (int)($_POST['user_id'] ?? 0);
        $message = trim($_POST['message'] ?? '');
        
        if ($userId <= 0 || empty($message)) {
            $error = 'Invalid user ID or empty message.';
        } else {
            try {
                // Get user's Telegram ID
                $stmt = $db->prepare("SELECT telegram_id, first_name FROM users WHERE id = ?");
                $stmt->execute([$userId]);
                $user = $stmt->fetch();
                
                if (!$user) {
                    $error = 'User not found.';
                } else {
                    // Send message via Telegram Bot
                    require_once '../classes/TelegramBot.php';
                    $bot = new TelegramBot(BOT_TOKEN);
                    $result = $bot->sendMessage($user['telegram_id'], $message);
                    
                    if ($result) {
                        $success = 'Message sent successfully to ' . htmlspecialchars($user['first_name']) . '.';
                        logActivity('admin', $userId, $_SESSION['admin_id'], 'send_message', "Sent message to user ID: $userId", $_SERVER['REMOTE_ADDR']);
                    } else {
                        $error = 'Error sending message. Please try again.';
                    }
                }
            } catch (Exception $e) {
                $error = 'Error sending message. Please try again.';
                logError("Send message error: " . $e->getMessage());
            }
        }
    }
}

// Get users based on filter
try {
    $query = "SELECT u.*, 
              (SELECT COUNT(*) FROM activity_logs WHERE user_id = u.id) as activity_count,
              (SELECT MAX(created_at) FROM activity_logs WHERE user_id = u.id) as last_activity
              FROM users u ";
    
    $params = [];
    $whereAdded = false;
    
    if (!empty($status)) {
        $query .= "WHERE u.status = ? ";
        $params[] = $status;
        $whereAdded = true;
    }
    
    if (!empty($search)) {
        if ($whereAdded) {
            $query .= "AND (u.first_name LIKE ? OR u.last_name LIKE ? OR u.username LIKE ?) ";
        } else {
            $query .= "WHERE (u.first_name LIKE ? OR u.last_name LIKE ? OR u.username LIKE ?) ";
            $whereAdded = true;
        }
        $searchParam = "%$search%";
        $params[] = $searchParam;
        $params[] = $searchParam;
        $params[] = $searchParam;
    }
    
    $query .= "ORDER BY u.created_at DESC";
    
    if (!empty($params)) {
        $stmt = $db->prepare($query);
        $stmt->execute($params);
    } else {
        $stmt = $db->query($query);
    }
    
    $users = $stmt->fetchAll();
} catch (Exception $e) {
    $error = 'Error loading users. Please try again.';
    logError("Load users error: " . $e->getMessage());
}

// Get user statistics
try {
    $stats = [
        'total' => 0,
        'active' => 0,
        'blocked' => 0,
        'today' => 0
    ];
    
    $stmt = $db->query("SELECT COUNT(*) as total FROM users");
    $stats['total'] = $stmt->fetch()['total'];
    
    $stmt = $db->query("SELECT COUNT(*) as active FROM users WHERE status = 'active'");
    $stats['active'] = $stmt->fetch()['active'];
    
    $stmt = $db->query("SELECT COUNT(*) as blocked FROM users WHERE status = 'blocked'");
    $stats['blocked'] = $stmt->fetch()['blocked'];
    
    $stmt = $db->query("SELECT COUNT(*) as today FROM users WHERE DATE(created_at) = CURDATE()");
    $stats['today'] = $stmt->fetch()['today'];
} catch (Exception $e) {
    logError("Load user stats error: " . $e->getMessage());
}

// Page title
$pageTitle = "User Management";

// Include header
include 'includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1 class="h3"><?php echo htmlspecialchars($pageTitle); ?></h1>
    </div>
    
    <?php if ($error): ?>
        <div class="alert alert-danger alert-auto-dismiss"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <?php if ($success): ?>
        <div class="alert alert-success alert-auto-dismiss"><?php echo htmlspecialchars($success); ?></div>
    <?php endif; ?>
    
    <!-- User Statistics -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-primary shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">Total Users</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $stats['total']; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-users fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-success shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">Active Users</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $stats['active']; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-user-check fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-warning shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">Blocked Users</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $stats['blocked']; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-user-slash fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-info shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">New Today</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $stats['today']; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-user-plus fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Filter Form -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">Filter Users</h6>
        </div>
        <div class="card-body">
            <form method="get" action="" class="row g-3">
                <div class="col-md-4">
                    <label for="status" class="form-label">Status</label>
                    <select class="form-select" id="status" name="status">
                        <option value="">All Users</option>
                        <option value="active" <?php echo $status === 'active' ? 'selected' : ''; ?>>Active</option>
                        <option value="blocked" <?php echo $status === 'blocked' ? 'selected' : ''; ?>>Blocked</option>
                    </select>
                </div>
                <div class="col-md-6">
                    <label for="search" class="form-label">Search</label>
                    <input type="text" class="form-control" id="search" name="search" value="<?php echo htmlspecialchars($search); ?>" placeholder="Search by name or username">
                </div>
                <div class="col-md-2 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary">Filter</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Quick Filter Buttons -->
    <div class="mb-4">
        <a href="users.php" class="btn btn-outline-primary me-2 <?php echo empty($status) && empty($search) ? 'active' : ''; ?>">
            All Users
        </a>
        <a href="users.php?status=active" class="btn btn-outline-success me-2 <?php echo $status === 'active' ? 'active' : ''; ?>">
            <i class="fas fa-user-check"></i> Active Users
        </a>
        <a href="users.php?status=blocked" class="btn btn-outline-danger <?php echo $status === 'blocked' ? 'active' : ''; ?>">
            <i class="fas fa-user-slash"></i> Blocked Users
        </a>
    </div>
    
    <!-- Users Table -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">
                <?php 
                if ($status === 'active') {
                    echo "Active Users";
                } elseif ($status === 'blocked') {
                    echo "Blocked Users";
                } else {
                    echo "All Users";
                }
                
                if (!empty($search)) {
                    echo " - Search: " . htmlspecialchars($search);
                }
                ?>
            </h6>
        </div>
        <div class="card-body">
            <?php if (empty($users)): ?>
                <p class="text-center text-muted">No users found.</p>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-bordered table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Telegram ID</th>
                                <th>Name</th>
                                <th>Username</th>
                                <th>Status</th>
                                <th>Activity</th>
                                <th>Last Active</th>
                                <th>Joined</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($users as $user): ?>
                                <tr>
                                    <td><?php echo $user['id']; ?></td>
                                    <td><?php echo $user['telegram_id']; ?></td>
                                    <td>
                                        <?php 
                                        $name = trim($user['first_name'] . ' ' . $user['last_name']);
                                        echo htmlspecialchars($name);
                                        ?>
                                    </td>
                                    <td>
                                        <?php if (!empty($user['username'])): ?>
                                            @<?php echo htmlspecialchars($user['username']); ?>
                                        <?php else: ?>
                                            <span class="text-muted">No username</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($user['status'] === 'active'): ?>
                                            <span class="badge bg-success">Active</span>
                                        <?php elseif ($user['status'] === 'blocked'): ?>
                                            <span class="badge bg-danger">Blocked</span>
                                        <?php else: ?>
                                            <span class="badge bg-secondary"><?php echo htmlspecialchars($user['status']); ?></span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo $user['activity_count']; ?> actions</td>
                                    <td>
                                        <?php if ($user['last_activity']): ?>
                                            <?php echo date('M d, Y H:i', strtotime($user['last_activity'])); ?>
                                        <?php else: ?>
                                            <span class="text-muted">Never</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo date('M d, Y H:i', strtotime($user['created_at'])); ?></td>
                                    <td>
                                        <?php if ($user['status'] === 'active'): ?>
                                            <button type="button" class="btn btn-sm btn-warning block-user"
                                                    data-id="<?php echo $user['id']; ?>"
                                                    data-name="<?php echo htmlspecialchars($name); ?>"
                                                    data-bs-toggle="modal" data-bs-target="#blockUserModal">
                                                <i class="fas fa-ban"></i> Block
                                            </button>
                                        <?php else: ?>
                                            <button type="button" class="btn btn-sm btn-success unblock-user"
                                                    data-id="<?php echo $user['id']; ?>"
                                                    data-name="<?php echo htmlspecialchars($name); ?>"
                                                    data-bs-toggle="modal" data-bs-target="#unblockUserModal">
                                                <i class="fas fa-check"></i> Unblock
                                            </button>
                                        <?php endif; ?>
                                        
                                        <button type="button" class="btn btn-sm btn-primary message-user"
                                                data-id="<?php echo $user['id']; ?>"
                                                data-name="<?php echo htmlspecialchars($name); ?>"
                                                data-bs-toggle="modal" data-bs-target="#messageUserModal">
                                            <i class="fas fa-envelope"></i> Message
                                        </button>
                                        
                                        <a href="user_activity.php?user_id=<?php echo $user['id']; ?>" class="btn btn-sm btn-info">
                                            <i class="fas fa-history"></i> Activity
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Block User Modal -->
<div class="modal fade" id="blockUserModal" tabindex="-1" aria-labelledby="blockUserModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="blockUserModalLabel">Block User</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to block <strong id="block-user-name"></strong>?</p>
                <p>Blocked users will not be able to use the bot.</p>
            </div>
            <form method="post" action="">
                <input type="hidden" name="action" value="block">
                <input type="hidden" name="user_id" id="block-user-id">
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-warning">Block User</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Unblock User Modal -->
<div class="modal fade" id="unblockUserModal" tabindex="-1" aria-labelledby="unblockUserModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="unblockUserModalLabel">Unblock User</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to unblock <strong id="unblock-user-name"></strong>?</p>
                <p>This will allow the user to use the bot again.</p>
            </div>
            <form method="post" action="">
                <input type="hidden" name="action" value="unblock">
                <input type="hidden" name="user_id" id="unblock-user-id">
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-success">Unblock User</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Message User Modal -->
<div class="modal fade" id="messageUserModal" tabindex="-1" aria-labelledby="messageUserModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="messageUserModalLabel">Send Message</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="post" action="">
                <div class="modal-body">
                    <input type="hidden" name="action" value="message">
                    <input type="hidden" name="user_id" id="message-user-id">
                    
                    <div class="mb-3">
                        <label for="message-recipient" class="form-label">Recipient</label>
                        <input type="text" class="form-control" id="message-recipient" readonly>
                    </div>
                    
                    <div class="mb-3">
                        <label for="message" class="form-label">Message</label>
                        <textarea class="form-control" id="message" name="message" rows="5" required></textarea>
                        <small class="text-muted">This message will be sent directly to the user via the bot.</small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Send Message</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php
// Add page-specific scripts
$pageScripts = ['../assets/js/users.js'];

// Include footer
include 'includes/footer.php';
?>