/**
 * Broadcast page JavaScript
 */

document.addEventListener('DOMContentLoaded', function() {
    // Initialize DataTable for broadcasts history
    if (document.getElementById('broadcastsTable')) {
        $('#broadcastsTable').DataTable({
            order: [[0, 'desc']], // Sort by date descending
            pageLength: 10,
            lengthMenu: [[5, 10, 25, 50, -1], [5, 10, 25, 50, "All"]],
            responsive: true
        });
    }
    
    // Character counter for message textarea
    const messageTextarea = document.getElementById('message');
    if (messageTextarea) {
        // Create character counter element
        const counterDiv = document.createElement('div');
        counterDiv.className = 'text-muted small mt-1';
        counterDiv.id = 'messageCounter';
        messageTextarea.parentNode.insertBefore(counterDiv, messageTextarea.nextSibling);
        
        // Update counter function
        const updateCounter = function() {
            const length = messageTextarea.value.length;
            const maxLength = 4096; // Telegram's max message length
            const remaining = maxLength - length;
            
            counterDiv.textContent = `${length} characters (${remaining} remaining)`;
            
            // Add warning if approaching limit
            if (remaining < 100) {
                counterDiv.className = 'text-danger small mt-1';
            } else if (remaining < 500) {
                counterDiv.className = 'text-warning small mt-1';
            } else {
                counterDiv.className = 'text-muted small mt-1';
            }
        };
        
        // Initial count
        updateCounter();
        
        // Update on input
        messageTextarea.addEventListener('input', updateCounter);
    }
    
    // Confirmation dialog for broadcast submission
    const broadcastForm = document.querySelector('form[action=""]');
    if (broadcastForm) {
        broadcastForm.addEventListener('submit', function(e) {
            if (!confirm('Are you sure you want to send this broadcast to all active users?')) {
                e.preventDefault();
                return false;
            }
            
            // Show loading state
            const submitButton = this.querySelector('button[type="submit"]');
            const originalText = submitButton.innerHTML;
            submitButton.disabled = true;
            submitButton.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Sending...';
            
            // Allow form submission
            return true;
        });
    }
    
    // Format helper buttons
    const formatButtons = document.getElementById('formatButtons');
    if (formatButtons && messageTextarea) {
        // Bold button
        document.getElementById('formatBold').addEventListener('click', function() {
            insertFormatting('<b>', '</b>');
        });
        
        // Italic button
        document.getElementById('formatItalic').addEventListener('click', function() {
            insertFormatting('<i>', '</i>');
        });
        
        // Code button
        document.getElementById('formatCode').addEventListener('click', function() {
            insertFormatting('<code>', '</code>');
        });
        
        // Pre button (preformatted text)
        document.getElementById('formatPre').addEventListener('click', function() {
            insertFormatting('<pre>', '</pre>');
        });
        
        // Link button
        document.getElementById('formatLink').addEventListener('click', function() {
            const url = prompt('Enter URL:', 'https://');
            if (url) {
                const text = prompt('Enter link text:', 'Link text');
                if (text) {
                    insertText(`<a href="${url}">${text}</a>`);
                }
            }
        });
        
        // Helper function to insert formatting tags
        function insertFormatting(openTag, closeTag) {
            const start = messageTextarea.selectionStart;
            const end = messageTextarea.selectionEnd;
            const selectedText = messageTextarea.value.substring(start, end);
            const replacement = openTag + selectedText + closeTag;
            
            messageTextarea.value = messageTextarea.value.substring(0, start) + 
                                   replacement + 
                                   messageTextarea.value.substring(end);
            
            // Update counter
            messageTextarea.dispatchEvent(new Event('input'));
            
            // Set cursor position after insertion
            messageTextarea.focus();
            if (selectedText.length > 0) {
                messageTextarea.selectionStart = start + replacement.length;
                messageTextarea.selectionEnd = start + replacement.length;
            } else {
                messageTextarea.selectionStart = start + openTag.length;
                messageTextarea.selectionEnd = start + openTag.length;
            }
        }
        
        // Helper function to insert text at cursor position
        function insertText(text) {
            const start = messageTextarea.selectionStart;
            messageTextarea.value = messageTextarea.value.substring(0, start) + 
                                   text + 
                                   messageTextarea.value.substring(start);
            
            // Update counter
            messageTextarea.dispatchEvent(new Event('input'));
            
            // Set cursor position after insertion
            messageTextarea.focus();
            messageTextarea.selectionStart = start + text.length;
            messageTextarea.selectionEnd = start + text.length;
        }
    }
});