<?php
/**
 * Admin Panel 500 Error Diagnostic Script
 * 
 * This script checks for common causes of 500 Internal Server Error in the admin panel
 * and provides suggestions for fixing them.
 */

// Enable error reporting for diagnostic purposes
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set content type to HTML
header('Content-Type: text/html; charset=utf-8');

// Start HTML output
echo '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Panel 500 Error Diagnostic</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            margin: 20px;
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        h1, h2, h3 {
            color: #333;
        }
        .success {
            color: green;
            background-color: #e8f5e9;
            padding: 10px;
            border-left: 5px solid green;
            margin-bottom: 10px;
        }
        .warning {
            color: #856404;
            background-color: #fff3cd;
            padding: 10px;
            border-left: 5px solid #ffeeba;
            margin-bottom: 10px;
        }
        .error {
            color: #721c24;
            background-color: #f8d7da;
            padding: 10px;
            border-left: 5px solid #f5c6cb;
            margin-bottom: 10px;
        }
        .info {
            color: #0c5460;
            background-color: #d1ecf1;
            padding: 10px;
            border-left: 5px solid #bee5eb;
            margin-bottom: 10px;
        }
        pre {
            background-color: #f5f5f5;
            padding: 10px;
            border-radius: 5px;
            overflow-x: auto;
        }
        code {
            font-family: monospace;
            background-color: #f5f5f5;
            padding: 2px 5px;
            border-radius: 3px;
        }
        .section {
            margin-bottom: 30px;
            border: 1px solid #ddd;
            padding: 20px;
            border-radius: 5px;
        }
        table {
            border-collapse: collapse;
            width: 100%;
            margin-bottom: 20px;
        }
        th, td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: left;
        }
        th {
            background-color: #f2f2f2;
        }
        .fix-button {
            background-color: #4CAF50;
            color: white;
            padding: 10px 15px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 16px;
            margin-top: 10px;
        }
        .fix-button:hover {
            background-color: #45a049;
        }
    </style>
</head>
<body>
    <h1>Admin Panel 500 Error Diagnostic</h1>
    <p>This script checks for common causes of 500 Internal Server Error in the admin panel.</p>
';

// Initialize variables
$issues = [];
$fixes = [];
$errors = [];
$warnings = [];
$success = [];

// Function to check if a directory exists and is writable
function checkDirectory($dir, $create = false) {
    if (!file_exists($dir)) {
        if ($create) {
            if (mkdir($dir, 0755, true)) {
                echo "<div class='success'>✅ Created directory: {$dir}</div>";
                return true;
            } else {
                echo "<div class='error'>❌ Failed to create directory: {$dir}</div>";
                return false;
            }
        } else {
            echo "<div class='error'>❌ Directory does not exist: {$dir}</div>";
            return false;
        }
    } elseif (!is_writable($dir)) {
        echo "<div class='error'>❌ Directory exists but is not writable: {$dir}</div>";
        return false;
    } else {
        echo "<div class='success'>✅ Directory exists and is writable: {$dir}</div>";
        return true;
    }
}

// Function to check if a file exists and is readable
function checkFile($file) {
    if (!file_exists($file)) {
        echo "<div class='error'>❌ File does not exist: {$file}</div>";
        return false;
    } elseif (!is_readable($file)) {
        echo "<div class='error'>❌ File exists but is not readable: {$file}</div>";
        return false;
    } else {
        echo "<div class='success'>✅ File exists and is readable: {$file}</div>";
        return true;
    }
}

// Function to check PHP extension
function checkExtension($extension) {
    if (extension_loaded($extension)) {
        echo "<div class='success'>✅ PHP extension loaded: {$extension}</div>";
        return true;
    } else {
        echo "<div class='error'>❌ PHP extension not loaded: {$extension}</div>";
        return false;
    }
}

// Function to check database connection
function checkDatabase() {
    global $issues, $success;
    
    echo "<div class='section'><h2>Database Connection Check</h2>";
    
    // Check if config file exists
    $configFile = dirname(__DIR__) . '/config/config.php';
    if (!file_exists($configFile)) {
        echo "<div class='error'>❌ Configuration file not found: {$configFile}</div>";
        $issues[] = "Configuration file not found";
        echo "</div>";
        return false;
    }
    
    // Include configuration
    require_once $configFile;
    
    // Check if database constants are defined
    if (!defined('DB_HOST') || !defined('DB_NAME') || !defined('DB_USER') || !defined('DB_PASS')) {
        echo "<div class='error'>❌ Database configuration constants not defined in config file</div>";
        $issues[] = "Database configuration constants not defined";
        echo "</div>";
        return false;
    }
    
    // Try to connect to database
    try {
        $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ];
        
        $db = new PDO($dsn, DB_USER, DB_PASS, $options);
        echo "<div class='success'>✅ Successfully connected to database</div>";
        
        // Check if required tables exist
        $requiredTables = ['admin_users', 'admin_sessions', 'settings', 'activity_logs'];
        $missingTables = [];
        
        foreach ($requiredTables as $table) {
            $stmt = $db->query("SHOW TABLES LIKE '{$table}'")->fetchColumn();
            if (!$stmt) {
                $missingTables[] = $table;
                echo "<div class='error'>❌ Required table not found: {$table}</div>";
            } else {
                echo "<div class='success'>✅ Required table exists: {$table}</div>";
            }
        }
        
        if (!empty($missingTables)) {
            $issues[] = "Missing required database tables: " . implode(', ', $missingTables);
        } else {
            $success[] = "All required database tables exist";
        }
        
        echo "</div>";
        return true;
        
    } catch (PDOException $e) {
        echo "<div class='error'>❌ Database connection failed: " . $e->getMessage() . "</div>";
        $issues[] = "Database connection failed: " . $e->getMessage();
        echo "</div>";
        return false;
    }
}

// Function to check logs directory
function checkLogs() {
    global $issues, $fixes, $success;
    
    echo "<div class='section'><h2>Logs Directory Check</h2>";
    
    $logsDir = dirname(__DIR__) . '/logs';
    $logsExists = checkDirectory($logsDir, false);
    
    if (!$logsExists) {
        $issues[] = "Logs directory does not exist or is not writable";
        $fixes[] = "Create logs directory with proper permissions";
    } else {
        $success[] = "Logs directory exists and is writable";
    }
    
    echo "</div>";
    return $logsExists;
}

// Function to check .htaccess file
function checkHtaccess() {
    global $issues, $fixes, $success;
    
    echo "<div class='section'><h2>.htaccess File Check</h2>";
    
    $htaccessFile = dirname(__DIR__) . '/admin/.htaccess';
    $htaccessExists = file_exists($htaccessFile);
    
    if (!$htaccessExists) {
        echo "<div class='error'>❌ .htaccess file does not exist in admin directory</div>";
        $issues[] = ".htaccess file does not exist in admin directory";
        $fixes[] = "Create .htaccess file in admin directory";
    } else {
        echo "<div class='success'>✅ .htaccess file exists in admin directory</div>";
        
        // Check .htaccess content
        $htaccessContent = file_get_contents($htaccessFile);
        
        if (strpos($htaccessContent, 'RewriteEngine On') === false) {
            echo "<div class='warning'>⚠️ .htaccess file does not have URL rewriting enabled</div>";
            $issues[] = ".htaccess file does not have URL rewriting enabled";
            $fixes[] = "Update .htaccess file with proper rewrite rules";
        } else {
            echo "<div class='success'>✅ .htaccess file has URL rewriting enabled</div>";
            $success[] = ".htaccess file is properly configured";
        }
    }
    
    echo "</div>";
    return $htaccessExists;
}

// Function to check PHP requirements
function checkPhpRequirements() {
    global $issues, $warnings, $success;
    
    echo "<div class='section'><h2>PHP Requirements Check</h2>";
    
    // Check PHP version
    $requiredPhpVersion = '7.2.0';
    $currentPhpVersion = phpversion();
    
    if (version_compare($currentPhpVersion, $requiredPhpVersion, '>=')) {
        echo "<div class='success'>✅ PHP version {$currentPhpVersion} meets requirement (>= {$requiredPhpVersion})</div>";
        $success[] = "PHP version {$currentPhpVersion} meets requirement";
    } else {
        echo "<div class='error'>❌ PHP version {$currentPhpVersion} does not meet requirement (>= {$requiredPhpVersion})</div>";
        $issues[] = "PHP version {$currentPhpVersion} does not meet requirement (>= {$requiredPhpVersion})";
    }
    
    // Check required extensions
    $requiredExtensions = ['pdo', 'pdo_mysql', 'mbstring', 'json', 'session'];
    $missingExtensions = [];
    
    foreach ($requiredExtensions as $extension) {
        if (!checkExtension($extension)) {
            $missingExtensions[] = $extension;
        }
    }
    
    if (!empty($missingExtensions)) {
        $issues[] = "Missing required PHP extensions: " . implode(', ', $missingExtensions);
    } else {
        $success[] = "All required PHP extensions are loaded";
    }
    
    echo "</div>";
}

// Function to check session configuration
function checkSessionConfig() {
    global $issues, $warnings, $success;
    
    echo "<div class='section'><h2>Session Configuration Check</h2>";
    
    // Check if SESSION_LIFETIME is defined
    if (!defined('SESSION_LIFETIME')) {
        echo "<div class='warning'>⚠️ SESSION_LIFETIME constant is not defined</div>";
        $warnings[] = "SESSION_LIFETIME constant is not defined";
    } else {
        echo "<div class='success'>✅ SESSION_LIFETIME constant is defined: " . SESSION_LIFETIME . " seconds</div>";
        $success[] = "SESSION_LIFETIME constant is properly defined";
    }
    
    // Check session save path
    $sessionSavePath = ini_get('session.save_path');
    if (empty($sessionSavePath)) {
        echo "<div class='warning'>⚠️ Session save path is not set</div>";
        $warnings[] = "Session save path is not set";
    } else {
        if (is_dir($sessionSavePath) && is_writable($sessionSavePath)) {
            echo "<div class='success'>✅ Session save path exists and is writable: {$sessionSavePath}</div>";
            $success[] = "Session save path is properly configured";
        } else {
            echo "<div class='error'>❌ Session save path does not exist or is not writable: {$sessionSavePath}</div>";
            $issues[] = "Session save path does not exist or is not writable";
        }
    }
    
    echo "</div>";
}

// Function to check error logging
function checkErrorLogging() {
    global $issues, $warnings, $success;
    
    echo "<div class='section'><h2>Error Logging Check</h2>";
    
    // Check if error logging is enabled
    $displayErrors = ini_get('display_errors');
    $logErrors = ini_get('log_errors');
    $errorLog = ini_get('error_log');
    
    echo "<div class='info'>ℹ️ display_errors = {$displayErrors}</div>";
    echo "<div class='info'>ℹ️ log_errors = {$logErrors}</div>";
    echo "<div class='info'>ℹ️ error_log = {$errorLog}</div>";
    
    if ($logErrors == '1') {
        echo "<div class='success'>✅ Error logging is enabled</div>";
        $success[] = "Error logging is enabled";
        
        if (!empty($errorLog)) {
            $errorLogDir = dirname($errorLog);
            if (is_dir($errorLogDir) && is_writable($errorLogDir)) {
                echo "<div class='success'>✅ Error log directory exists and is writable</div>";
                $success[] = "Error log directory is writable";
            } else {
                echo "<div class='error'>❌ Error log directory does not exist or is not writable: {$errorLogDir}</div>";
                $issues[] = "Error log directory does not exist or is not writable";
            }
        } else {
            echo "<div class='warning'>⚠️ Error log path is not set</div>";
            $warnings[] = "Error log path is not set";
        }
    } else {
        echo "<div class='warning'>⚠️ Error logging is not enabled</div>";
        $warnings[] = "Error logging is not enabled";
    }
    
    // Check if logError function exists
    if (function_exists('logError')) {
        echo "<div class='success'>✅ logError function exists</div>";
        $success[] = "logError function exists";
    } else {
        echo "<div class='error'>❌ logError function does not exist</div>";
        $issues[] = "logError function does not exist";
    }
    
    echo "</div>";
}

// Function to check admin directory structure
function checkAdminStructure() {
    global $issues, $warnings, $success;
    
    echo "<div class='section'><h2>Admin Directory Structure Check</h2>";
    
    $adminDir = dirname(__DIR__) . '/admin';
    $adminExists = is_dir($adminDir);
    
    if (!$adminExists) {
        echo "<div class='error'>❌ Admin directory does not exist: {$adminDir}</div>";
        $issues[] = "Admin directory does not exist";
        echo "</div>";
        return false;
    }
    
    echo "<div class='success'>✅ Admin directory exists: {$adminDir}</div>";
    
    // Check required files
    $requiredFiles = [
        'index.php',
        'login.php',
        'auth.php',
        'includes/header.php',
        'includes/footer.php'
    ];
    
    $missingFiles = [];
    foreach ($requiredFiles as $file) {
        $filePath = $adminDir . '/' . $file;
        if (!file_exists($filePath)) {
            echo "<div class='error'>❌ Required file does not exist: {$file}</div>";
            $missingFiles[] = $file;
        } else {
            echo "<div class='success'>✅ Required file exists: {$file}</div>";
        }
    }
    
    if (!empty($missingFiles)) {
        $issues[] = "Missing required admin files: " . implode(', ', $missingFiles);
    } else {
        $success[] = "All required admin files exist";
    }
    
    echo "</div>";
    return true;
}

// Function to fix common issues
function fixCommonIssues() {
    global $fixes;
    
    echo "<div class='section'><h2>Applying Fixes</h2>";
    
    // Create logs directory if it doesn't exist
    $logsDir = dirname(__DIR__) . '/logs';
    if (!file_exists($logsDir)) {
        if (mkdir($logsDir, 0755, true)) {
            echo "<div class='success'>✅ Created logs directory: {$logsDir}</div>";
            $fixes[] = "Created logs directory";
        } else {
            echo "<div class='error'>❌ Failed to create logs directory: {$logsDir}</div>";
        }
    }
    
    // Create .htaccess file if it doesn't exist
    $htaccessFile = dirname(__DIR__) . '/admin/.htaccess';
    if (!file_exists($htaccessFile)) {
        $htaccessContent = "# Enable URL rewriting\nRewriteEngine On\n\n# Set the base for rewriting\nRewriteBase /new_bot/admin/\n\n# Prevent direct access to PHP files in includes directory\nRewriteRule ^includes/ - [F,L]\n\n# Handle 404 errors\nErrorDocument 404 /new_bot/admin/index.php\n\n# Handle 500 errors\nErrorDocument 500 /new_bot/admin/error.php\n\n# Ensure PHP errors are displayed\nphp_flag display_errors On\nphp_value error_reporting E_ALL\n\n# Protect against XSS attacks\n<IfModule mod_headers.c>\n    Header set X-XSS-Protection \"1; mode=block\"\n    Header set X-Content-Type-Options \"nosniff\"\n    Header set X-Frame-Options \"SAMEORIGIN\"\n</IfModule>\n\n# Disable directory listing\nOptions -Indexes";
        
        if (file_put_contents($htaccessFile, $htaccessContent)) {
            echo "<div class='success'>✅ Created .htaccess file in admin directory</div>";
            $fixes[] = "Created .htaccess file in admin directory";
        } else {
            echo "<div class='error'>❌ Failed to create .htaccess file in admin directory</div>";
        }
    }
    
    // Create error.php file for handling 500 errors
    $errorFile = dirname(__DIR__) . '/admin/error.php';
    if (!file_exists($errorFile)) {
        $errorContent = "<?php\n/**\n * Error page for 500 Internal Server Error\n */\n\n// Enable error reporting for debugging\nerror_reporting(E_ALL);\nini_set('display_errors', 1);\n\n// Set content type to HTML\nheader('Content-Type: text/html; charset=utf-8');\n?>\n<!DOCTYPE html>\n<html lang=\"en\">\n<head>\n    <meta charset=\"UTF-8\">\n    <meta name=\"viewport\" content=\"width=device-width, initial-scale=1.0\">\n    <title>500 Internal Server Error</title>\n    <style>\n        body {\n            font-family: Arial, sans-serif;\n            line-height: 1.6;\n            margin: 20px;\n            max-width: 800px;\n            margin: 0 auto;\n            padding: 20px;\n        }\n        h1 {\n            color: #d32f2f;\n        }\n        .error-box {\n            background-color: #ffebee;\n            border-left: 5px solid #d32f2f;\n            padding: 15px;\n            margin-bottom: 20px;\n        }\n        .info-box {\n            background-color: #e3f2fd;\n            border-left: 5px solid #2196f3;\n            padding: 15px;\n            margin-bottom: 20px;\n        }\n        pre {\n            background-color: #f5f5f5;\n            padding: 10px;\n            border-radius: 5px;\n            overflow-x: auto;\n        }\n        .button {\n            display: inline-block;\n            background-color: #2196f3;\n            color: white;\n            padding: 10px 15px;\n            text-decoration: none;\n            border-radius: 4px;\n            margin-top: 20px;\n        }\n        .button:hover {\n            background-color: #0b7dda;\n        }\n    </style>\n</head>\n<body>\n    <h1>500 Internal Server Error</h1>\n    \n    <div class=\"error-box\">\n        <p>The server encountered an internal error and was unable to complete your request.</p>\n        <p>Please try the following:</p>\n        <ul>\n            <li>Refresh the page</li>\n            <li>Clear your browser cache</li>\n            <li>Try again later</li>\n        </ul>\n    </div>\n    \n    <div class=\"info-box\">\n        <p>If you are the administrator, please check the following:</p>\n        <ul>\n            <li>PHP error logs</li>\n            <li>Database connection</li>\n            <li>File permissions</li>\n            <li>Run the diagnostic script: <code>/check_admin_500_error.php</code></li>\n        </ul>\n    </div>\n    \n    <a href=\"../index.php\" class=\"button\">Go to Homepage</a>\n    <a href=\"login.php\" class=\"button\">Go to Login</a>\n</body>\n</html>";
        
        if (file_put_contents($errorFile, $errorContent)) {
            echo "<div class='success'>✅ Created error.php file in admin directory</div>";
            $fixes[] = "Created error.php file in admin directory";
        } else {
            echo "<div class='error'>❌ Failed to create error.php file in admin directory</div>";
        }
    }
    
    echo "</div>";
}

// Run checks
echo "<div class='section'><h2>Running Diagnostics</h2>";
echo "<p>PHP Version: " . phpversion() . "</p>";
echo "<p>Server Software: " . $_SERVER['SERVER_SOFTWARE'] . "</p>";
echo "<p>Document Root: " . $_SERVER['DOCUMENT_ROOT'] . "</p>";
echo "</div>";

// Check database connection
checkDatabase();

// Check logs directory
checkLogs();

// Check .htaccess file
checkHtaccess();

// Check PHP requirements
checkPhpRequirements();

// Check session configuration
checkSessionConfig();

// Check error logging
checkErrorLogging();

// Check admin directory structure
checkAdminStructure();

// Apply fixes if there are issues
if (!empty($issues)) {
    echo "<div class='section'><h2>Issues Found</h2>";
    echo "<ul>";
    foreach ($issues as $issue) {
        echo "<li>{$issue}</li>";
    }
    echo "</ul>";
    echo "<button class='fix-button' onclick='location.href="?fix=1"'>Apply Fixes</button>";
    echo "</div>";
    
    if (isset($_GET['fix']) && $_GET['fix'] == '1') {
        fixCommonIssues();
    }
} else {
    echo "<div class='section'><h2>No Issues Found</h2>";
    echo "<div class='success'>✅ All checks passed. If you're still experiencing 500 errors, please check the server error logs for more details.</div>";
    echo "</div>";
}

// Summary
echo "<div class='section'><h2>Summary</h2>";

if (!empty($success)) {
    echo "<h3>Successful Checks:</h3>";
    echo "<ul>";
    foreach ($success as $item) {
        echo "<li>{$item}</li>";
    }
    echo "</ul>";
}

if (!empty($warnings)) {
    echo "<h3>Warnings:</h3>";
    echo "<ul>";
    foreach ($warnings as $warning) {
        echo "<li>{$warning}</li>";
    }
    echo "</ul>";
}

if (!empty($issues)) {
    echo "<h3>Issues:</h3>";
    echo "<ul>";
    foreach ($issues as $issue) {
        echo "<li>{$issue}</li>";
    }
    echo "</ul>";
}

if (!empty($fixes)) {
    echo "<h3>Fixes Applied:</h3>";
    echo "<ul>";
    foreach ($fixes as $fix) {
        echo "<li>{$fix}</li>";
    }
    echo "</ul>";
}

echo "</div>";

// End HTML output
echo '</body>
</html>';