<?php
/**
 * Admin Panel Accessibility Check
 * 
 * This script checks if the admin panel is accessible and properly configured.
 */

// Include configuration
require_once __DIR__ . '/config/config.php';

echo "=== Admin Panel Accessibility Check ===\n\n";

// Check if admin panel URL is configured
if (empty(ADMIN_PANEL_URL)) {
    echo "❌ Admin panel URL is not configured in config.php\n";
    exit(1);
}

// Function to check URL accessibility
function checkUrl($url) {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_NOBODY, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    return [
        'code' => $httpCode,
        'error' => $error
    ];
}

// Check admin panel URL
echo "Checking admin panel URL: " . ADMIN_PANEL_URL . "\n";
$result = checkUrl(ADMIN_PANEL_URL);

if ($result['code'] >= 200 && $result['code'] < 300) {
    echo "✅ Admin panel is accessible (HTTP {$result['code']})\n";
} elseif ($result['code'] >= 300 && $result['code'] < 400) {
    echo "⚠️ Admin panel URL is redirecting (HTTP {$result['code']})\n";
    echo "   This might be normal if you have URL rewriting configured.\n";
} elseif ($result['code'] == 404) {
    echo "❌ Admin panel URL returns 404 Not Found\n";
    echo "   Possible issues:\n";
    echo "   1. The URL path is incorrect\n";
    echo "   2. The admin directory doesn't exist on the server\n";
    echo "   3. .htaccess file is missing or not configured properly\n";
    echo "\n   Recommended actions:\n";
    echo "   1. Verify that the admin directory exists at the correct location\n";
    echo "   2. Check if the server supports .htaccess files\n";
    echo "   3. Make sure mod_rewrite is enabled on the server\n";
} elseif ($result['code'] == 403) {
    echo "❌ Admin panel URL returns 403 Forbidden\n";
    echo "   Possible issues:\n";
    echo "   1. Directory permissions are too restrictive\n";
    echo "   2. .htaccess is blocking access\n";
} elseif ($result['code'] == 500) {
    echo "❌ Admin panel URL returns 500 Internal Server Error\n";
    echo "   Possible issues:\n";
    echo "   1. PHP error in the admin panel code\n";
    echo "   2. Database connection issue\n";
    echo "   3. Server configuration problem\n";
} elseif (empty($result['code']) && !empty($result['error'])) {
    echo "❌ Error connecting to admin panel: {$result['error']}\n";
} else {
    echo "❌ Admin panel URL returns unexpected status code: {$result['code']}\n";
}

// Check admin login page
echo "\nChecking admin login page...\n";
$loginUrl = rtrim(ADMIN_PANEL_URL, '/') . '/login.php';
echo "URL: {$loginUrl}\n";

$result = checkUrl($loginUrl);

if ($result['code'] >= 200 && $result['code'] < 300) {
    echo "✅ Admin login page is accessible (HTTP {$result['code']})\n";
} elseif ($result['code'] >= 300 && $result['code'] < 400) {
    echo "⚠️ Admin login URL is redirecting (HTTP {$result['code']})\n";
} elseif ($result['code'] == 404) {
    echo "❌ Admin login page returns 404 Not Found\n";
} else {
    echo "❌ Admin login page returns status code: {$result['code']}\n";
}

// Check database tables
echo "\nChecking database tables...\n";

try {
    // Create database connection
    $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME;
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    $pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
    
    // Check admin_users table
    $stmt = $pdo->query("SHOW TABLES LIKE 'admin_users'");
    if ($stmt->rowCount() > 0) {
        echo "✅ admin_users table exists\n";
        
        // Check if default admin user exists
        $stmt = $pdo->query("SELECT COUNT(*) as count FROM admin_users WHERE username = 'admin'");
        $result = $stmt->fetch();
        if ($result['count'] > 0) {
            echo "✅ Default admin user exists\n";
        } else {
            echo "❌ Default admin user does not exist\n";
        }
    } else {
        echo "❌ admin_users table does not exist\n";
    }
    
    // Check admin_sessions table
    $stmt = $pdo->query("SHOW TABLES LIKE 'admin_sessions'");
    if ($stmt->rowCount() > 0) {
        echo "✅ admin_sessions table exists\n";
    } else {
        echo "❌ admin_sessions table does not exist\n";
    }
    
} catch (PDOException $e) {
    echo "❌ Database connection error: " . $e->getMessage() . "\n";
}

echo "\n=== Check Complete ===\n";