<?php
/**
 * Check Bot Status Script
 * 
 * This script checks the status and information of your Telegram bot.
 * It verifies the bot token, gets bot information, and checks webhook status.
 */

// Include configuration
require_once __DIR__ . '/config/config.php';

echo "=== Checking Telegram Bot Status ===\n\n";

// Validate configuration
if (empty(BOT_TOKEN)) {
    die("Error: Bot token is not configured. Please update your config.php file.\n");
}

// Get bot information
echo "Getting bot information...\n";
$url = "https://api.telegram.org/bot" . BOT_TOKEN . "/getMe";

if (function_exists('curl_version')) {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    $response = curl_exec($ch);
    curl_close($ch);
} else {
    $response = file_get_contents($url);
}

$result = json_decode($response, true);

if (isset($result['ok']) && $result['ok']) {
    echo "\nBot Information:\n";
    echo "- ID: {$result['result']['id']}\n";
    echo "- Name: {$result['result']['first_name']}\n";
    echo "- Username: @{$result['result']['username']}\n";
    echo "- Can join groups: " . ($result['result']['can_join_groups'] ? "Yes" : "No") . "\n";
    echo "- Can read all group messages: " . ($result['result']['can_read_all_group_messages'] ? "Yes" : "No") . "\n";
    echo "- Supports inline queries: " . ($result['result']['supports_inline_queries'] ? "Yes" : "No") . "\n";
    
    // Check if the configured username matches the actual username
    if ($result['result']['username'] !== BOT_USERNAME) {
        echo "\nWarning: Configured username (@" . BOT_USERNAME . ") does not match actual bot username (@{$result['result']['username']})\n";
        echo "Please update the BOT_USERNAME in your config.php file.\n";
    }
} else {
    echo "\nFailed to get bot information!\n";
    echo "Error: " . (isset($result['description']) ? $result['description'] : "Unknown error") . "\n";
    echo "Please check your bot token.\n";
    exit(1);
}

// Get webhook information
echo "\nGetting webhook information...\n";
$url = "https://api.telegram.org/bot" . BOT_TOKEN . "/getWebhookInfo";

if (function_exists('curl_version')) {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    $response = curl_exec($ch);
    curl_close($ch);
} else {
    $response = file_get_contents($url);
}

$result = json_decode($response, true);

if (isset($result['ok']) && $result['ok']) {
    echo "\nWebhook Information:\n";
    echo "- URL: {$result['result']['url']}\n";
    echo "- Has custom certificate: " . ($result['result']['has_custom_certificate'] ? "Yes" : "No") . "\n";
    echo "- Pending update count: {$result['result']['pending_update_count']}\n";
    
    if (isset($result['result']['last_error_date'])) {
        $date = date('Y-m-d H:i:s', $result['result']['last_error_date']);
        echo "- Last error date: {$date}\n";
        echo "- Last error message: {$result['result']['last_error_message']}\n";
    }
    
    if (isset($result['result']['max_connections'])) {
        echo "- Max connections: {$result['result']['max_connections']}\n";
    }
    
    if (isset($result['result']['allowed_updates']) && !empty($result['result']['allowed_updates'])) {
        echo "- Allowed updates: " . implode(", ", $result['result']['allowed_updates']) . "\n";
    }
    
    // Check if the webhook URL matches the configured webhook URL
    if ($result['result']['url'] !== WEBHOOK_URL) {
        echo "\nWarning: Current webhook URL ({$result['result']['url']}) does not match configured webhook URL (" . WEBHOOK_URL . ")\n";
        echo "You may need to update your webhook. Run set_webhook.php to update it.\n";
    }
    
    // Check for webhook errors
    if (isset($result['result']['last_error_message'])) {
        echo "\nWarning: Webhook has errors. Last error: {$result['result']['last_error_message']}\n";
        echo "Please check your server configuration and ensure the webhook URL is accessible.\n";
    }
} else {
    echo "\nFailed to get webhook information!\n";
    echo "Error: " . (isset($result['description']) ? $result['description'] : "Unknown error") . "\n";
}

// Get bot commands
echo "\nGetting bot commands...\n";
$url = "https://api.telegram.org/bot" . BOT_TOKEN . "/getMyCommands";

if (function_exists('curl_version')) {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    $response = curl_exec($ch);
    curl_close($ch);
} else {
    $response = file_get_contents($url);
}

$result = json_decode($response, true);

if (isset($result['ok']) && $result['ok']) {
    if (!empty($result['result'])) {
        echo "\nBot Commands:\n";
        foreach ($result['result'] as $command) {
            echo "- /{$command['command']} - {$command['description']}\n";
        }
    } else {
        echo "\nNo commands are set for this bot.\n";
        echo "You can set commands using the /setcommands command with BotFather or programmatically.\n";
    }
} else {
    echo "\nFailed to get bot commands!\n";
    echo "Error: " . (isset($result['description']) ? $result['description'] : "Unknown error") . "\n";
}

echo "\n=== Bot Check Complete ===\n";