<?php
/**
 * Database Connection Check
 * 
 * This script checks if the database connection is working properly.
 */

// Include configuration
require_once __DIR__ . '/config/config.php';

echo "=== Database Connection Check ===\n\n";

// Check database credentials
echo "Database credentials:\n";
echo "Host: " . DB_HOST . "\n";
echo "Name: " . DB_NAME . "\n";
echo "User: " . DB_USER . "\n";
echo "Pass: " . str_repeat('*', strlen(DB_PASS)) . "\n\n";

try {
    // Create database connection
    $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME;
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    echo "Connecting to database...\n";
    $pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
    echo "✅ Connected successfully!\n\n";
    
    // Check database tables
    echo "Checking database tables...\n";
    $stmt = $pdo->query("SHOW TABLES");
    $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    if (count($tables) > 0) {
        echo "Found " . count($tables) . " tables:\n";
        foreach ($tables as $table) {
            echo "- {$table}\n";
        }
    } else {
        echo "❌ No tables found in the database.\n";
        echo "You may need to run the database initialization script:\n";
        echo "php database/init_db.php\n";
    }
    
    // Check required tables
    $requiredTables = [
        'users',
        'admin_users',
        'admin_sessions',
        'menus',
        'submenus',
        'content',
        'settings',
        'activity_logs'
    ];
    
    echo "\nChecking required tables...\n";
    $missingTables = [];
    
    foreach ($requiredTables as $table) {
        if (!in_array($table, $tables)) {
            $missingTables[] = $table;
        }
    }
    
    if (count($missingTables) > 0) {
        echo "❌ Missing tables:\n";
        foreach ($missingTables as $table) {
            echo "- {$table}\n";
        }
        echo "\nYou should run the database initialization script:\n";
        echo "php database/init_db.php\n";
    } else {
        echo "✅ All required tables exist!\n";
    }
    
} catch (PDOException $e) {
    echo "❌ Database connection error: " . $e->getMessage() . "\n";
    
    // Provide more specific error messages
    if (strpos($e->getMessage(), "Access denied") !== false) {
        echo "\nPossible issues:\n";
        echo "1. Incorrect username or password\n";
        echo "2. User does not have access to the database\n";
        echo "\nSolution: Check your database credentials in config.php\n";
    } elseif (strpos($e->getMessage(), "Unknown database") !== false) {
        echo "\nPossible issues:\n";
        echo "1. Database does not exist\n";
        echo "2. Database name is misspelled\n";
        echo "\nSolution: Create the database or check the database name in config.php\n";
    } elseif (strpos($e->getMessage(), "Could not find driver") !== false) {
        echo "\nPossible issues:\n";
        echo "1. PDO MySQL extension is not installed or enabled\n";
        echo "\nSolution: Install or enable the PDO MySQL extension\n";
    } elseif (strpos($e->getMessage(), "Connection refused") !== false) {
        echo "\nPossible issues:\n";
        echo "1. MySQL server is not running\n";
        echo "2. MySQL server is not accepting connections\n";
        echo "3. Firewall is blocking the connection\n";
        echo "\nSolution: Start the MySQL server or check firewall settings\n";
    }
}

echo "\n=== Check Complete ===\n";