<?php
/**
 * TelegramBot Class
 * Handles all interactions with the Telegram Bot API
 */

class TelegramBot {
    private $token;
    private $apiUrl = 'https://api.telegram.org/bot';
    private $db;
    private $debug = false;
    
    /**
     * Constructor
     * @param string $token Bot token
     * @param PDO $db Database connection
     * @param bool $debug Enable debug mode
     */
    public function __construct($token, $db, $debug = false) {
        $this->token = $token;
        $this->db = $db;
        $this->debug = $debug;
        $this->apiUrl .= $token . '/';
    }
    
    /**
     * Send message to a chat
     * @param int $chatId Chat ID
     * @param string $text Message text
     * @param array $options Additional options
     * @return array|bool Response from API or false on failure
     */
    public function sendMessage($chatId, $text, $options = []) {
        if ($this->debug) {
            $this->logDebug("Sending message to $chatId: $text");
        }
        
        // Format text if markdown is enabled
        if (isset($options['parse_mode']) && $options['parse_mode'] == 'Markdown') {
            $text = $this->formatMessageText($text);
        }
        
        $data = [
            'chat_id' => $chatId,
            'text' => $text
        ];
        
        // Merge additional options
        if (!empty($options)) {
            $data = array_merge($data, $options);
        }
        
        return $this->makeRequest('sendMessage', $data);
    }
    
    /**
     * Edit a message
     * @param int $chatId Chat ID
     * @param int $messageId Message ID
     * @param string $text New message text
     * @param array $options Additional options
     * @return array|bool Response from API or false on failure
     */
    public function editMessageText($chatId, $messageId, $text, $options = []) {
        $data = [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $text
        ];
        
        // Merge additional options
        if (!empty($options)) {
            $data = array_merge($data, $options);
        }
        
        return $this->makeRequest('editMessageText', $data);
    }
    
    /**
     * Send photo to a chat
     * @param int $chatId Chat ID
     * @param string $photo Photo path or file_id
     * @param string $caption Caption text
     * @param array $options Additional options
     * @return array|bool Response from API or false on failure
     */
    public function sendPhoto($chatId, $photo, $caption = '', $options = []) {
        $data = [
            'chat_id' => $chatId,
            'photo' => $photo
        ];
        
        if (!empty($caption)) {
            $data['caption'] = $caption;
        }
        
        // Merge additional options
        if (!empty($options)) {
            $data = array_merge($data, $options);
        }
        
        return $this->makeRequest('sendPhoto', $data);
    }
    
    /**
     * Send document to a chat
     * @param int $chatId Chat ID
     * @param string $document Document path or file_id
     * @param string $caption Caption text
     * @param array $options Additional options
     * @return array|bool Response from API or false on failure
     */
    public function sendDocument($chatId, $document, $caption = '', $options = []) {
        $data = [
            'chat_id' => $chatId,
            'document' => $document
        ];
        
        if (!empty($caption)) {
            $data['caption'] = $caption;
        }
        
        // Merge additional options
        if (!empty($options)) {
            $data = array_merge($data, $options);
        }
        
        return $this->makeRequest('sendDocument', $data);
    }
    
    /**
     * Set webhook URL
     * @param string $url Webhook URL
     * @param array $options Additional options
     * @return array|bool Response from API or false on failure
     */
    public function setWebhook($url, $options = []) {
        $data = ['url' => $url];
        
        // Merge additional options
        if (!empty($options)) {
            $data = array_merge($data, $options);
        }
        
        return $this->makeRequest('setWebhook', $data);
    }
    
    /**
     * Delete webhook
     * @return array|bool Response from API or false on failure
     */
    public function deleteWebhook() {
        return $this->makeRequest('deleteWebhook');
    }
    
    /**
     * Get webhook info
     * @return array|bool Response from API or false on failure
     */
    public function getWebhookInfo() {
        return $this->makeRequest('getWebhookInfo');
    }
    
    /**
     * Answer callback query
     * @param string $callbackQueryId Callback query ID
     * @param string $text Text to show
     * @param bool $showAlert Show as alert
     * @return array|bool Response from API or false on failure
     */
    public function answerCallbackQuery($callbackQueryId, $text = '', $showAlert = false) {
        $data = ['callback_query_id' => $callbackQueryId];
        
        if (!empty($text)) {
            $data['text'] = $text;
        }
        
        if ($showAlert) {
            $data['show_alert'] = true;
        }
        
        return $this->makeRequest('answerCallbackQuery', $data);
    }
    
    /**
     * Get bot info
     * @return array|bool Response from API or false on failure
     */
    public function getMe() {
        return $this->makeRequest('getMe');
    }
    
    /**
     * Format message text for Markdown
     * @param string $text Text to format
     * @return string Formatted text
     */
    private function formatMessageText($text) {
        // Convert Markdown to HTML entities to avoid parsing errors
        $text = preg_replace('/\\([\[\]()~`>#+=|{}.!-])/', '$1', $text);
        
        if ($this->debug) {
            $this->logDebug("Formatted message: $text");
        }
        
        return $text;
    }
    
    /**
     * Make request to Telegram API
     * @param string $method API method
     * @param array $data Request data
     * @return array|bool Response from API or false on failure
     */
    private function makeRequest($method, $data = []) {
        $url = $this->apiUrl . $method;
        
        $options = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_TIMEOUT => 30,
        ];
        
        // If data is provided, add it to the request
        if (!empty($data)) {
            $options[CURLOPT_POSTFIELDS] = $data;
        }
        
        $curl = curl_init();
        curl_setopt_array($curl, $options);
        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $error = curl_error($curl);
        curl_close($curl);
        
        // Log request details if debug is enabled
        if ($this->debug) {
            $this->logDebug("API Request: $method");
            $this->logDebug("API Data: " . json_encode($data));
            $this->logDebug("API Response Code: $httpCode");
            $this->logDebug("API Response: $response");
            if ($error) {
                $this->logDebug("API Error: $error");
            }
        }
        
        // Check for cURL errors
        if ($error) {
            logError("cURL Error: $error");
            return false;
        }
        
        // Check for HTTP errors
        if ($httpCode >= 400) {
            logError("HTTP Error $httpCode: $response");
            return false;
        }
        
        // Decode response
        $responseData = json_decode($response, true);
        
        // Check for API errors
        if (!$responseData['ok']) {
            $errorCode = isset($responseData['error_code']) ? $responseData['error_code'] : 'unknown';
            $errorDesc = isset($responseData['description']) ? $responseData['description'] : 'No description';
            logError("API Error $errorCode: $errorDesc");
            return false;
        }
        
        return $responseData['result'];
    }
    
    /**
     * Save user to database
     * @param array $user User data from Telegram
     * @return bool Success status
     */
    public function saveUser($user) {
        try {
            // Check if user exists
            $stmt = $this->db->prepare("SELECT id FROM users WHERE telegram_id = ?");
            $stmt->execute([$user['id']]);
            $existingUser = $stmt->fetch();
            
            if ($existingUser) {
                // Update existing user
                $stmt = $this->db->prepare("UPDATE users SET 
                    first_name = ?, 
                    last_name = ?, 
                    username = ?, 
                    language_code = ?, 
                    last_interaction = NOW() 
                    WHERE telegram_id = ?");
                $stmt->execute([
                    $user['first_name'] ?? '',
                    $user['last_name'] ?? '',
                    $user['username'] ?? '',
                    $user['language_code'] ?? 'en',
                    $user['id']
                ]);
            } else {
                // Insert new user
                $stmt = $this->db->prepare("INSERT INTO users 
                    (telegram_id, first_name, last_name, username, language_code, is_admin) 
                    VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->execute([
                    $user['id'],
                    $user['first_name'] ?? '',
                    $user['last_name'] ?? '',
                    $user['username'] ?? '',
                    $user['language_code'] ?? 'en',
                    isAdmin($user['id']) ? 1 : 0
                ]);
            }
            
            return true;
        } catch (Exception $e) {
            logError("Failed to save user: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get active menus
     * @return array List of menus
     */
    public function getMenus() {
        try {
            $stmt = $this->db->prepare("SELECT * FROM menus WHERE is_active = 1 ORDER BY sort_order ASC");
            $stmt->execute();
            return $stmt->fetchAll();
        } catch (Exception $e) {
            logError("Failed to get menus: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get submenus for a menu
     * @param int $menuId Menu ID
     * @return array List of submenus
     */
    public function getSubmenus($menuId) {
        try {
            $stmt = $this->db->prepare("SELECT * FROM submenus WHERE menu_id = ? AND is_active = 1 ORDER BY sort_order ASC");
            $stmt->execute([$menuId]);
            return $stmt->fetchAll();
        } catch (Exception $e) {
            logError("Failed to get submenus: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get content for a submenu
     * @param int $submenuId Submenu ID
     * @return array List of content items
     */
    public function getContent($submenuId) {
        try {
            $stmt = $this->db->prepare("SELECT * FROM content WHERE submenu_id = ? AND is_active = 1");
            $stmt->execute([$submenuId]);
            return $stmt->fetchAll();
        } catch (Exception $e) {
            logError("Failed to get content: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get links for a submenu
     * @param int $submenuId Submenu ID
     * @return array List of links
     */
    public function getLinks($submenuId) {
        try {
            $stmt = $this->db->prepare("SELECT * FROM links WHERE submenu_id = ? AND is_active = 1");
            $stmt->execute([$submenuId]);
            return $stmt->fetchAll();
        } catch (Exception $e) {
            logError("Failed to get links: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Create inline keyboard markup
     * @param array $buttons Array of button rows
     * @return array Inline keyboard markup
     */
    public function createInlineKeyboard($buttons) {
        return ['inline_keyboard' => $buttons];
    }
    
    /**
     * Create keyboard markup
     * @param array $buttons Array of button rows
     * @param bool $resizeKeyboard Resize keyboard
     * @param bool $oneTimeKeyboard One time keyboard
     * @return array Keyboard markup
     */
    public function createKeyboard($buttons, $resizeKeyboard = true, $oneTimeKeyboard = false) {
        return [
            'keyboard' => $buttons,
            'resize_keyboard' => $resizeKeyboard,
            'one_time_keyboard' => $oneTimeKeyboard
        ];
    }
    
    /**
     * Log debug message
     * @param string $message Debug message
     */
    private function logDebug($message) {
        if ($this->debug) {
            $logFile = dirname(__DIR__) . '/logs/debug.log';
            $logMessage = date('[Y-m-d H:i:s]') . ' DEBUG: ' . $message . PHP_EOL;
            file_put_contents($logFile, $logMessage, FILE_APPEND);
        }
    }
}