<?php
/**
 * Create Admin Structure Script
 * 
 * This script creates the necessary directory structure for the admin panel.
 */

// Include configuration
require_once __DIR__ . '/config/config.php';

echo "=== Creating Admin Panel Structure ===\n\n";

// Define directories to create
$directories = [
    __DIR__ . '/admin',
    __DIR__ . '/admin/includes',
    __DIR__ . '/assets/css',
    __DIR__ . '/assets/js',
    __DIR__ . '/assets/img'
];

// Create directories
foreach ($directories as $dir) {
    if (!file_exists($dir)) {
        echo "Creating directory: {$dir}\n";
        if (mkdir($dir, 0755, true)) {
            echo "✅ Directory created successfully\n";
        } else {
            echo "❌ Failed to create directory\n";
        }
    } else {
        echo "Directory already exists: {$dir}\n";
    }
}

// Create .htaccess file if it doesn't exist
$htaccessFile = __DIR__ . '/admin/.htaccess';
if (!file_exists($htaccessFile)) {
    echo "\nCreating .htaccess file...\n";
    $htaccessContent = "# Enable URL rewriting\nRewriteEngine On\n\n# Set the base for rewriting\nRewriteBase /new_bot/admin/\n\n# Prevent direct access to PHP files in includes directory\nRewriteRule ^includes/ - [F,L]\n\n# Handle 404 errors\nErrorDocument 404 /new_bot/admin/index.php\n\n# Ensure PHP errors are displayed\nphp_flag display_errors On\nphp_value error_reporting E_ALL\n\n# Protect against XSS attacks\n<IfModule mod_headers.c>\n    Header set X-XSS-Protection \"1; mode=block\"\n    Header set X-Content-Type-Options \"nosniff\"\n    Header set X-Frame-Options \"SAMEORIGIN\"\n</IfModule>\n\n# Disable directory listing\nOptions -Indexes\n";
    
    if (file_put_contents($htaccessFile, $htaccessContent)) {
        echo "✅ .htaccess file created successfully\n";
    } else {
        echo "❌ Failed to create .htaccess file\n";
    }
} else {
    echo "\n.htaccess file already exists\n";
}

// Create index.html file if it doesn't exist
$indexHtmlFile = __DIR__ . '/admin/index.html';
if (!file_exists($indexHtmlFile)) {
    echo "\nCreating index.html file...\n";
    $indexHtmlContent = "<!DOCTYPE html>\n<html lang=\"en\">\n<head>\n    <meta charset=\"UTF-8\">\n    <meta http-equiv=\"refresh\" content=\"0;url=index.php\">\n    <title>Redirecting...</title>\n</head>\n<body>\n    <p>If you are not redirected automatically, follow this <a href=\"index.php\">link to the admin panel</a>.</p>\n</body>\n</html>";
    
    if (file_put_contents($indexHtmlFile, $indexHtmlContent)) {
        echo "✅ index.html file created successfully\n";
    } else {
        echo "❌ Failed to create index.html file\n";
    }
} else {
    echo "\nindex.html file already exists\n";
}

// Create README.md file if it doesn't exist
$readmeFile = __DIR__ . '/admin/README.md';
if (!file_exists($readmeFile)) {
    echo "\nCreating README.md file...\n";
    $readmeContent = "# Telegram Bot Admin Panel\n\n## Overview\n\nThis directory contains the admin panel for managing your Telegram bot. The admin panel provides a web interface to:\n\n- View bot statistics and user activity\n- Manage menus and submenus\n- Create and edit content\n- Send broadcast messages to users\n- Configure bot settings\n\n## Files and Structure\n\n- `index.php` - Dashboard with statistics and recent activity\n- `login.php` - Admin login page\n- `auth.php` - Authentication handler\n- `menus.php` - Menu management\n- `submenus.php` - Submenu management\n- `content.php` - Content management\n- `broadcast.php` - Send messages to users\n- `settings.php` - Bot configuration\n- `users.php` - User management\n- `logout.php` - Logout handler\n- `includes/` - Common includes and templates\n\n## Access\n\nAccess the admin panel at: " . ADMIN_PANEL_URL . "\n\nDefault login credentials:\n- Username: admin\n- Password: admin123\n\n**Important**: Change the default password immediately after logging in!\n\n## Troubleshooting\n\nIf you encounter a 404 Not Found error when accessing the admin panel, run the fix script:\n\n```bash\nphp ../fix_admin_tables.php\n```\n\nThis will ensure all required database tables exist.\n\n## Security\n\nThe admin panel includes several security features:\n\n- Session-based authentication\n- Password hashing\n- CSRF protection\n- Session expiration\n- IP and user agent tracking\n\nFor additional security, consider:\n\n1. Changing the default admin username and password\n2. Enabling HTTPS\n3. Implementing two-factor authentication\n4. Restricting access by IP address\n";
    
    if (file_put_contents($readmeFile, $readmeContent)) {
        echo "✅ README.md file created successfully\n";
    } else {
        echo "❌ Failed to create README.md file\n";
    }
} else {
    echo "\nREADME.md file already exists\n";
}

echo "\n=== Admin Panel Structure Creation Complete ===\n";
echo "You can now access the admin panel at: " . ADMIN_PANEL_URL . "\n";