<?php
/**
 * Create Telegram Bot Admin Database
 * 
 * This script helps create the telegram_bot_admin database and grant proper permissions.
 */

// Enable error reporting for diagnostic purposes
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set content type to HTML
header('Content-Type: text/html; charset=utf-8');

// Start HTML output
echo '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Telegram Bot Admin Database</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            margin: 20px;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }
        h1, h2, h3 {
            color: #333;
        }
        .success {
            color: green;
            background-color: #e8f5e9;
            padding: 10px;
            border-left: 5px solid green;
            margin-bottom: 10px;
        }
        .warning {
            color: #856404;
            background-color: #fff3cd;
            padding: 10px;
            border-left: 5px solid #ffeeba;
            margin-bottom: 10px;
        }
        .error {
            color: #721c24;
            background-color: #f8d7da;
            padding: 10px;
            border-left: 5px solid #f5c6cb;
            margin-bottom: 10px;
        }
        .info {
            color: #0c5460;
            background-color: #d1ecf1;
            padding: 10px;
            border-left: 5px solid #bee5eb;
            margin-bottom: 10px;
        }
        pre {
            background-color: #f5f5f5;
            padding: 10px;
            border-radius: 5px;
            overflow-x: auto;
        }
        code {
            font-family: monospace;
            background-color: #f5f5f5;
            padding: 2px 5px;
            border-radius: 3px;
        }
        .section {
            margin-bottom: 30px;
            border: 1px solid #ddd;
            padding: 20px;
            border-radius: 5px;
        }
        form {
            margin-bottom: 20px;
        }
        input[type=text], input[type=password] {
            width: 100%;
            padding: 12px 20px;
            margin: 8px 0;
            display: inline-block;
            border: 1px solid #ccc;
            border-radius: 4px;
            box-sizing: border-box;
        }
        .form-group {
            margin-bottom: 15px;
        }
        label {
            font-weight: bold;
        }
        button {
            background-color: #4CAF50;
            color: white;
            padding: 10px 15px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 16px;
        }
        button:hover {
            background-color: #45a049;
        }
    </style>
</head>
<body>
    <h1>Create Telegram Bot Admin Database</h1>
    <p>This tool helps you create the telegram_bot_admin database and grant proper permissions.</p>
';

// Function to get current config settings
function getCurrentConfig() {
    $configFile = dirname(__DIR__) . '/config/config.php';
    
    if (!file_exists($configFile)) {
        echo "<div class='error'>❌ Configuration file not found: {$configFile}</div>";
        return false;
    }
    
    // Include configuration file
    include_once $configFile;
    
    // Check if database constants are defined
    if (!defined('DB_HOST') || !defined('DB_NAME') || !defined('DB_USER') || !defined('DB_PASS')) {
        echo "<div class='error'>❌ Database configuration constants not defined in config file</div>";
        return false;
    }
    
    return [
        'host' => DB_HOST,
        'name' => DB_NAME,
        'user' => DB_USER,
        'pass' => DB_PASS
    ];
}

// Function to test database connection
function testDatabaseConnection($host, $user, $pass, $name = null) {
    try {
        $dsn = "mysql:host={$host}" . ($name ? ";dbname={$name}" : '');
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ];
        
        $db = new PDO($dsn, $user, $pass, $options);
        return ['success' => true, 'message' => 'Connection successful', 'connection' => $db];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Function to check if database exists
function checkDatabaseExists($host, $user, $pass, $name) {
    try {
        $dsn = "mysql:host={$host}";
        $db = new PDO($dsn, $user, $pass);
        
        $stmt = $db->query("SELECT SCHEMA_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = '{$name}'");
        $result = $stmt->fetchColumn();
        
        return ['success' => true, 'exists' => !empty($result)];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Function to create database
function createDatabase($host, $user, $pass, $name) {
    try {
        $dsn = "mysql:host={$host}";
        $db = new PDO($dsn, $user, $pass);
        
        // Create database if it doesn't exist
        $db->exec("CREATE DATABASE IF NOT EXISTS `{$name}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        
        return ['success' => true, 'message' => "Database '{$name}' created successfully"];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Function to grant privileges
function grantPrivileges($host, $adminUser, $adminPass, $dbName, $dbUser) {
    try {
        $dsn = "mysql:host={$host}";
        $db = new PDO($dsn, $adminUser, $adminPass);
        
        // Grant privileges to user
        $db->exec("GRANT ALL PRIVILEGES ON `{$dbName}`.* TO '{$dbUser}'@'localhost'");
        $db->exec("FLUSH PRIVILEGES");
        
        return ['success' => true, 'message' => "Privileges granted to user '{$dbUser}' on database '{$dbName}'"];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Main execution
echo "<div class='section'>";

// Get configuration settings
$config = getCurrentConfig();

if ($config) {
    echo "<h2>Current Configuration</h2>";
    echo "<ul>";
    echo "<li><strong>Host:</strong> {$config['host']}</li>";
    echo "<li><strong>Database:</strong> {$config['name']}</li>";
    echo "<li><strong>User:</strong> {$config['user']}</li>";
    echo "<li><strong>Password:</strong> " . str_repeat('*', strlen($config['pass'])) . "</li>";
    echo "</ul>";
    
    // Check if telegram_bot_admin database exists
    $dbName = 'telegram_bot_admin';
    $dbExists = checkDatabaseExists($config['host'], $config['user'], $config['pass'], $dbName);
    
    if ($dbExists['success']) {
        if ($dbExists['exists']) {
            echo "<div class='info'>ℹ️ Database '{$dbName}' already exists</div>";
            
            // Test connection to the database
            $dbConnection = testDatabaseConnection($config['host'], $config['user'], $config['pass'], $dbName);
            
            if ($dbConnection['success']) {
                echo "<div class='success'>✅ Successfully connected to database '{$dbName}'</div>";
                echo "<p>Your user already has access to this database.</p>";
            } else {
                echo "<div class='error'>❌ Failed to connect to database '{$dbName}': {$dbConnection['message']}</div>";
                
                // Check if it's an access denied error
                if (strpos($dbConnection['message'], 'Access denied') !== false) {
                    echo "<div class='info'>ℹ️ This is an access denied error. The database exists but your user does not have access to it.</div>";
                    
                    // Display form to grant privileges
                    echo "<h3>Grant Privileges</h3>";
                    echo "<form method='post' action=''>";
                    echo "<input type='hidden' name='action' value='grant_privileges'>";
                    echo "<div class='form-group'>";
                    echo "<label for='admin_user'>MySQL Admin Username:</label>";
                    echo "<input type='text' id='admin_user' name='admin_user' required>";
                    echo "</div>";
                    echo "<div class='form-group'>";
                    echo "<label for='admin_pass'>MySQL Admin Password:</label>";
                    echo "<input type='password' id='admin_pass' name='admin_pass' required>";
                    echo "</div>";
                    echo "<button type='submit'>Grant Privileges</button>";
                    echo "</form>";
                }
            }
        } else {
            echo "<div class='info'>ℹ️ Database '{$dbName}' does not exist</div>";
            
            // Display form to create database
            echo "<h3>Create Database</h3>";
            echo "<form method='post' action=''>";
            echo "<input type='hidden' name='action' value='create_database'>";
            echo "<div class='form-group'>";
            echo "<label for='admin_user'>MySQL Admin Username:</label>";
            echo "<input type='text' id='admin_user' name='admin_user' required>";
            echo "</div>";
            echo "<div class='form-group'>";
            echo "<label for='admin_pass'>MySQL Admin Password:</label>";
            echo "<input type='password' id='admin_pass' name='admin_pass' required>";
            echo "</div>";
            echo "<button type='submit'>Create Database</button>";
            echo "</form>";
        }
    } else {
        echo "<div class='error'>❌ Failed to check if database exists: {$dbExists['message']}</div>";
        
        // Check if it's an access denied error
        if (strpos($dbExists['message'], 'Access denied') !== false) {
            echo "<div class='info'>ℹ️ This is an access denied error. Your MySQL user does not have sufficient privileges.</div>";
            
            // Display form to check with admin credentials
            echo "<h3>Check Database with Admin Credentials</h3>";
            echo "<form method='post' action=''>";
            echo "<input type='hidden' name='action' value='check_database'>";
            echo "<div class='form-group'>";
            echo "<label for='admin_user'>MySQL Admin Username:</label>";
            echo "<input type='text' id='admin_user' name='admin_user' required>";
            echo "</div>";
            echo "<div class='form-group'>";
            echo "<label for='admin_pass'>MySQL Admin Password:</label>";
            echo "<input type='password' id='admin_pass' name='admin_pass' required>";
            echo "</div>";
            echo "<button type='submit'>Check Database</button>";
            echo "</form>";
        }
    }
}

echo "</div>";

// Process form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    echo "<div class='section'><h2>Action Results</h2>";
    
    switch ($action) {
        case 'create_database':
            if (isset($_POST['admin_user']) && isset($_POST['admin_pass'])) {
                // Create database using admin credentials
                $result = createDatabase($config['host'], $_POST['admin_user'], $_POST['admin_pass'], 'telegram_bot_admin');
                
                if ($result['success']) {
                    echo "<div class='success'>✅ {$result['message']}</div>";
                    
                    // Grant privileges to the configured user
                    $grantResult = grantPrivileges($config['host'], $_POST['admin_user'], $_POST['admin_pass'], 'telegram_bot_admin', $config['user']);
                    
                    if ($grantResult['success']) {
                        echo "<div class='success'>✅ {$grantResult['message']}</div>";
                        echo "<p>You can now use the telegram_bot_admin database with your application.</p>";
                        
                        // Suggest updating config.php if needed
                        if ($config['name'] !== 'telegram_bot_admin') {
                            echo "<div class='warning'>⚠️ Your application is currently configured to use database '{$config['name']}', but you've created 'telegram_bot_admin'.</div>";
                            echo "<p>You may want to update your config.php file to use the new database.</p>";
                            echo "<p><a href='update_config_db.php'>Click here to update your configuration</a></p>";
                        }
                    } else {
                        echo "<div class='error'>❌ Failed to grant privileges: {$grantResult['message']}</div>";
                    }
                } else {
                    echo "<div class='error'>❌ Failed to create database: {$result['message']}</div>";
                }
            } else {
                echo "<div class='error'>❌ MySQL admin credentials are required</div>";
            }
            break;
            
        case 'grant_privileges':
            if (isset($_POST['admin_user']) && isset($_POST['admin_pass'])) {
                $result = grantPrivileges($config['host'], $_POST['admin_user'], $_POST['admin_pass'], 'telegram_bot_admin', $config['user']);
                
                if ($result['success']) {
                    echo "<div class='success'>✅ {$result['message']}</div>";
                    echo "<p>You can now use the telegram_bot_admin database with your application.</p>";
                    
                    // Suggest updating config.php if needed
                    if ($config['name'] !== 'telegram_bot_admin') {
                        echo "<div class='warning'>⚠️ Your application is currently configured to use database '{$config['name']}', but you've granted access to 'telegram_bot_admin'.</div>";
                        echo "<p>You may want to update your config.php file to use the new database.</p>";
                        echo "<p><a href='update_config_db.php'>Click here to update your configuration</a></p>";
                    }
                } else {
                    echo "<div class='error'>❌ Failed to grant privileges: {$result['message']}</div>";
                }
            } else {
                echo "<div class='error'>❌ MySQL admin credentials are required</div>";
            }
            break;
            
        case 'check_database':
            if (isset($_POST['admin_user']) && isset($_POST['admin_pass'])) {
                // Check if database exists using admin credentials
                $dbExists = checkDatabaseExists($config['host'], $_POST['admin_user'], $_POST['admin_pass'], 'telegram_bot_admin');
                
                if ($dbExists['success']) {
                    if ($dbExists['exists']) {
                        echo "<div class='success'>✅ Database 'telegram_bot_admin' exists</div>";
                        
                        // Test connection to the database with configured user
                        $dbConnection = testDatabaseConnection($config['host'], $config['user'], $config['pass'], 'telegram_bot_admin');
                        
                        if ($dbConnection['success']) {
                            echo "<div class='success'>✅ Your user can access the database</div>";
                        } else {
                            echo "<div class='error'>❌ Your user cannot access the database: {$dbConnection['message']}</div>";
                            
                            // Display form to grant privileges
                            echo "<h3>Grant Privileges</h3>";
                            echo "<form method='post' action=''>";
                            echo "<input type='hidden' name='action' value='grant_privileges'>";
                            echo "<input type='hidden' name='admin_user' value='{$_POST['admin_user']}'>";
                            echo "<input type='hidden' name='admin_pass' value='{$_POST['admin_pass']}'>";
                            echo "<button type='submit'>Grant Privileges to Your User</button>";
                            echo "</form>";
                        }
                    } else {
                        echo "<div class='info'>ℹ️ Database 'telegram_bot_admin' does not exist</div>";
                        
                        // Display form to create database
                        echo "<h3>Create Database</h3>";
                        echo "<form method='post' action=''>";
                        echo "<input type='hidden' name='action' value='create_database'>";
                        echo "<input type='hidden' name='admin_user' value='{$_POST['admin_user']}'>";
                        echo "<input type='hidden' name='admin_pass' value='{$_POST['admin_pass']}'>";
                        echo "<button type='submit'>Create Database</button>";
                        echo "</form>";
                    }
                } else {
                    echo "<div class='error'>❌ Failed to check if database exists: {$dbExists['message']}</div>";
                }
            } else {
                echo "<div class='error'>❌ MySQL admin credentials are required</div>";
            }
            break;
    }
    
    echo "<p><a href='?'>Back to main page</a></p>";
    echo "</div>";
}

// End HTML output
echo '</body>
</html>';