#!/bin/bash

# Deployment script for Telegram Bot with Admin Panel
# This script helps with deploying the bot to a production server

# Display help message
show_help() {
    echo "Telegram Bot Deployment Script"
    echo "Usage: ./deploy.sh [options]"
    echo ""
    echo "Options:"
    echo "  -h, --help       Show this help message"
    echo "  -i, --init       Initialize the project (create config, set permissions)"
    echo "  -d, --database   Initialize the database"
    echo "  -w, --webhook    Set up the webhook"
    echo "  -a, --all        Perform all initialization steps"
    echo "  -u, --update     Update an existing installation"
}

# Initialize project
init_project() {
    echo "Initializing project..."
    
    # Create config file if it doesn't exist
    if [ ! -f "config/config.php" ]; then
        echo "Creating config file..."
        cp config/config.sample.php config/config.php
        echo "Please edit config/config.php with your settings"
    else
        echo "Config file already exists"
    fi
    
    # Create logs directory if it doesn't exist
    if [ ! -d "logs" ]; then
        echo "Creating logs directory..."
        mkdir -p logs
    fi
    
    # Set permissions
    echo "Setting permissions..."
    chmod 755 .
    chmod 644 .htaccess
    chmod -R 755 admin
    chmod -R 644 admin/*.php admin/assets/*
    chmod -R 755 admin/assets
    chmod 755 webhook.php
    chmod -R 755 includes classes
    chmod 644 includes/*.php classes/*.php
    chmod -R 755 logs
    chmod 644 config/config.sample.php
    chmod 600 config/config.php
    
    echo "Project initialized successfully"
}

# Initialize database
init_database() {
    echo "Initializing database..."
    
    # Check if config file exists
    if [ ! -f "config/config.php" ]; then
        echo "Error: Config file not found. Run with --init first."
        exit 1
    fi
    
    # Run database initialization script
    php database/init_db.php
    
    echo "Database initialization completed"
}

# Set up webhook
setup_webhook() {
    echo "Setting up webhook..."
    
    # Check if config file exists
    if [ ! -f "config/config.php" ]; then
        echo "Error: Config file not found. Run with --init first."
        exit 1
    fi
    
    # Get webhook URL and bot token from config
    WEBHOOK_URL=$(grep -o "'webhook_url' => '[^']*'" config/config.php | cut -d "'" -f 4)
    BOT_TOKEN=$(grep -o "'token' => '[^']*'" config/config.php | cut -d "'" -f 4)
    
    if [ -z "$WEBHOOK_URL" ] || [ -z "$BOT_TOKEN" ]; then
        echo "Error: Webhook URL or Bot Token not found in config file."
        exit 1
    fi
    
    # Set webhook using Telegram API
    echo "Setting webhook to: $WEBHOOK_URL"
    curl -s "https://api.telegram.org/bot$BOT_TOKEN/setWebhook?url=$WEBHOOK_URL"
    echo ""
    
    echo "Webhook setup completed"
}

# Update existing installation
update_installation() {
    echo "Updating installation..."
    
    # Pull latest changes if git is used
    if [ -d ".git" ]; then
        echo "Pulling latest changes from git..."
        git pull
    else
        echo "Not a git repository. Manual update required."
    fi
    
    # Set permissions again
    echo "Updating permissions..."
    chmod 755 .
    chmod 644 .htaccess
    chmod -R 755 admin
    chmod -R 644 admin/*.php admin/assets/*
    chmod -R 755 admin/assets
    chmod 755 webhook.php
    chmod -R 755 includes classes
    chmod 644 includes/*.php classes/*.php
    chmod -R 755 logs
    
    echo "Update completed"
}

# Process command line arguments
if [ $# -eq 0 ]; then
    show_help
    exit 0
fi

while [ $# -gt 0 ]; do
    case "$1" in
        -h|--help)
            show_help
            exit 0
            ;;
        -i|--init)
            init_project
            ;;
        -d|--database)
            init_database
            ;;
        -w|--webhook)
            setup_webhook
            ;;
        -a|--all)
            init_project
            init_database
            setup_webhook
            ;;
        -u|--update)
            update_installation
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
    shift
done

exit 0