<?php
/**
 * 500 Error Fix Script
 * 
 * This script diagnoses and fixes common causes of 500 Internal Server Error
 * in the Telegram Bot Admin Panel.
 */

// Enable error reporting for diagnostic purposes
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set content type to HTML
header('Content-Type: text/html; charset=utf-8');

// Start HTML output
echo '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>500 Error Fix Tool</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            margin: 20px;
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        h1, h2, h3 {
            color: #333;
        }
        .success {
            color: green;
            background-color: #e8f5e9;
            padding: 10px;
            border-left: 5px solid green;
            margin-bottom: 10px;
        }
        .warning {
            color: #856404;
            background-color: #fff3cd;
            padding: 10px;
            border-left: 5px solid #ffeeba;
            margin-bottom: 10px;
        }
        .error {
            color: #721c24;
            background-color: #f8d7da;
            padding: 10px;
            border-left: 5px solid #f5c6cb;
            margin-bottom: 10px;
        }
        .info {
            color: #0c5460;
            background-color: #d1ecf1;
            padding: 10px;
            border-left: 5px solid #bee5eb;
            margin-bottom: 10px;
        }
        pre {
            background-color: #f5f5f5;
            padding: 10px;
            border-radius: 5px;
            overflow-x: auto;
        }
        code {
            font-family: monospace;
            background-color: #f5f5f5;
            padding: 2px 5px;
            border-radius: 3px;
        }
        .section {
            margin-bottom: 30px;
            border: 1px solid #ddd;
            padding: 20px;
            border-radius: 5px;
        }
        .fix-button {
            background-color: #4CAF50;
            color: white;
            padding: 10px 15px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            margin-top: 10px;
        }
        .fix-button:hover {
            background-color: #45a049;
        }
    </style>
</head>
<body>
    <h1>500 Error Fix Tool</h1>
    <p>This tool diagnoses and fixes common causes of 500 Internal Server Error in your Telegram Bot Admin Panel.</p>
    
    <div class="section">
        <h2>1. Checking Logs Directory</h2>
';

// Check logs directory
$logsDir = __DIR__ . '/logs';
$logsDirExists = file_exists($logsDir);
$logsDirWritable = false;

if (!$logsDirExists) {
    echo '<div class="warning">Logs directory does not exist. Creating it now...</div>';
    if (mkdir($logsDir, 0755, true)) {
        echo '<div class="success">Logs directory created successfully.</div>';
        $logsDirExists = true;
        $logsDirWritable = is_writable($logsDir);
    } else {
        echo '<div class="error">Failed to create logs directory. Please create it manually with proper permissions.</div>';
    }
} else {
    echo '<div class="success">Logs directory exists.</div>';
    $logsDirWritable = is_writable($logsDir);
    if (!$logsDirWritable) {
        echo '<div class="error">Logs directory is not writable. Please set proper permissions (chmod 755).</div>';
    } else {
        echo '<div class="success">Logs directory is writable.</div>';
    }
}

// Create test log file
if ($logsDirExists && $logsDirWritable) {
    $testLogFile = $logsDir . '/test.log';
    if (file_put_contents($testLogFile, date('[Y-m-d H:i:s]') . ' Test log entry' . PHP_EOL)) {
        echo '<div class="success">Successfully wrote to test log file.</div>';
    } else {
        echo '<div class="error">Failed to write to test log file.</div>';
    }
}

echo '</div>';

// Check for logError function
echo '<div class="section">
    <h2>2. Checking Logger Functionality</h2>';

// Check if Logger.php exists and is readable
$loggerFile = __DIR__ . '/includes/Logger.php';
if (file_exists($loggerFile) && is_readable($loggerFile)) {
    echo '<div class="success">Logger.php file exists and is readable.</div>';
    
    // Check if logError function exists
    if (!function_exists('logError')) {
        echo '<div class="warning">logError function not found. Creating it now...</div>';
        
        // Create logError function
        $logErrorFunction = <<<'EOD'
/**
 * Log an error message
 * @param string $message Error message
 * @return bool Success status
 */
function logError($message) {
    static $logger = null;
    
    if ($logger === null) {
        require_once __DIR__ . '/includes/Logger.php';
        $logger = new Logger();
    }
    
    return $logger->log($message, 'error');
}
EOD;
        
        // Append to functions.php
        $functionsFile = __DIR__ . '/includes/functions.php';
        if (file_exists($functionsFile) && is_writable($functionsFile)) {
            if (file_put_contents($functionsFile, PHP_EOL . $logErrorFunction . PHP_EOL, FILE_APPEND)) {
                echo '<div class="success">logError function added to functions.php.</div>';
            } else {
                echo '<div class="error">Failed to add logError function to functions.php.</div>';
            }
        } else {
            echo '<div class="error">functions.php file does not exist or is not writable.</div>';
        }
    } else {
        echo '<div class="success">logError function exists.</div>';
    }
} else {
    echo '<div class="error">Logger.php file does not exist or is not readable.</div>';
}

echo '</div>';

// Check database connection
echo '<div class="section">
    <h2>3. Testing Database Connection</h2>';

// Include configuration
$configFile = __DIR__ . '/config/config.php';
if (file_exists($configFile) && is_readable($configFile)) {
    echo '<div class="success">Config file exists and is readable.</div>';
    
    // Load configuration
    require_once $configFile;
    
    // Test database connection
    try {
        $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ];
        
        $db = new PDO($dsn, DB_USER, DB_PASS, $options);
        echo '<div class="success">Database connection successful.</div>';
        
        // Check if required tables exist
        $requiredTables = ['users', 'admin_users', 'menus', 'submenus', 'content', 'settings', 'activity_logs'];
        $missingTables = [];
        
        foreach ($requiredTables as $table) {
            $stmt = $db->query("SHOW TABLES LIKE '{$table}'");
            if ($stmt->rowCount() == 0) {
                $missingTables[] = $table;
            }
        }
        
        if (empty($missingTables)) {
            echo '<div class="success">All required tables exist.</div>';
        } else {
            echo '<div class="warning">Missing tables: ' . implode(', ', $missingTables) . '</div>';
            echo '<div class="info">You may need to import the database schema. Check the database/schema.sql file.</div>';
        }
        
    } catch (PDOException $e) {
        echo '<div class="error">Database connection failed: ' . htmlspecialchars($e->getMessage()) . '</div>';
        echo '<div class="info">Please check your database configuration in config.php.</div>';
    }
} else {
    echo '<div class="error">Config file does not exist or is not readable.</div>';
}

echo '</div>';

// Check .htaccess file
echo '<div class="section">
    <h2>4. Checking .htaccess Configuration</h2>';

$htaccessFile = __DIR__ . '/admin/.htaccess';
$htaccessContent = '';

if (file_exists($htaccessFile) && is_readable($htaccessFile)) {
    echo '<div class="success">.htaccess file exists and is readable.</div>';
    $htaccessContent = file_get_contents($htaccessFile);
    
    // Check for ErrorDocument directive
    if (strpos($htaccessContent, 'ErrorDocument 500') === false) {
        echo '<div class="warning">.htaccess file does not have 500 error handling. Adding it now...</div>';
        
        // Add ErrorDocument directive
        $errorDirective = "\n# Custom error pages\nErrorDocument 500 /new_bot/admin/error.php?code=500\n";
        
        if (file_put_contents($htaccessFile, $htaccessContent . $errorDirective)) {
            echo '<div class="success">Added 500 error handling to .htaccess file.</div>';
        } else {
            echo '<div class="error">Failed to update .htaccess file.</div>';
        }
    } else {
        echo '<div class="success">.htaccess file has 500 error handling.</div>';
    }
} else {
    echo '<div class="warning">.htaccess file does not exist or is not readable. Creating it now...</div>';
    
    // Create basic .htaccess file
    $htaccessContent = <<<'EOD'
# Protect directory
Options -Indexes

# Enable URL rewriting
RewriteEngine On

# Custom error pages
ErrorDocument 500 /new_bot/admin/error.php?code=500

# PHP settings
php_flag display_errors off
php_value error_reporting E_ALL

# Protect sensitive files
<FilesMatch "\.(htaccess|htpasswd|ini|log|sh|inc|bak)$">
Order Allow,Deny
Deny from all
</FilesMatch>

# Set security headers
<IfModule mod_headers.c>
Header set X-Content-Type-Options "nosniff"
Header set X-XSS-Protection "1; mode=block"
Header set X-Frame-Options "SAMEORIGIN"
</IfModule>
EOD;
    
    if (file_put_contents($htaccessFile, $htaccessContent)) {
        echo '<div class="success">Created .htaccess file with proper configuration.</div>';
    } else {
        echo '<div class="error">Failed to create .htaccess file.</div>';
    }
}

echo '</div>';

// Check for error.php file
echo '<div class="section">
    <h2>5. Checking Error Page</h2>';

$errorFile = __DIR__ . '/admin/error.php';

if (file_exists($errorFile) && is_readable($errorFile)) {
    echo '<div class="success">error.php file exists and is readable.</div>';
} else {
    echo '<div class="warning">error.php file does not exist or is not readable. Creating it now...</div>';
    
    // Create error.php file
    $errorContent = <<<'EOD'
<?php
/**
 * Error page for Admin Panel
 */

// Get error code
$errorCode = isset($_GET['code']) ? intval($_GET['code']) : 500;
$errorMessage = '';

// Set appropriate error message
switch ($errorCode) {
    case 400:
        $errorMessage = 'Bad Request';
        break;
    case 401:
        $errorMessage = 'Unauthorized';
        break;
    case 403:
        $errorMessage = 'Forbidden';
        break;
    case 404:
        $errorMessage = 'Page Not Found';
        break;
    case 500:
    default:
        $errorCode = 500; // Ensure we use 500 as default
        $errorMessage = 'Internal Server Error';
        break;
}

// Log the error if it's a server error
if ($errorCode >= 500 && function_exists('logError')) {
    logError("Error {$errorCode} occurred: {$errorMessage}");
}

// Set content type
header('Content-Type: text/html; charset=utf-8');

// Set HTTP response code
http_response_code($errorCode);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Error <?php echo $errorCode; ?> - Admin Panel</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            color: #333;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
            text-align: center;
        }
        .error-container {
            margin-top: 50px;
            border: 1px solid #ddd;
            padding: 20px;
            border-radius: 5px;
            background-color: #f9f9f9;
        }
        .error-code {
            font-size: 72px;
            margin: 0;
            color: #e74c3c;
        }
        .error-message {
            font-size: 24px;
            margin-top: 0;
        }
        .error-details {
            margin-top: 20px;
            color: #666;
        }
        .back-link {
            display: inline-block;
            margin-top: 20px;
            padding: 10px 15px;
            background-color: #3498db;
            color: white;
            text-decoration: none;
            border-radius: 4px;
        }
        .back-link:hover {
            background-color: #2980b9;
        }
    </style>
</head>
<body>
    <div class="error-container">
        <h1 class="error-code"><?php echo $errorCode; ?></h1>
        <h2 class="error-message"><?php echo $errorMessage; ?></h2>
        
        <div class="error-details">
            <?php if ($errorCode == 500): ?>
                <p>Sorry, something went wrong on our server.</p>
                <p>The error has been logged and we'll work on fixing it as soon as possible.</p>
            <?php elseif ($errorCode == 404): ?>
                <p>The page you are looking for might have been removed, had its name changed, or is temporarily unavailable.</p>
            <?php elseif ($errorCode == 403): ?>
                <p>You don't have permission to access this resource.</p>
            <?php elseif ($errorCode == 401): ?>
                <p>Authentication is required to access this resource.</p>
            <?php else: ?>
                <p>An error occurred while processing your request.</p>
            <?php endif; ?>
        </div>
        
        <a href="/new_bot/admin/" class="back-link">Back to Admin Panel</a>
    </div>
</body>
</html>
EOD;
    
    if (file_put_contents($errorFile, $errorContent)) {
        echo '<div class="success">Created error.php file with proper configuration.</div>';
    } else {
        echo '<div class="error">Failed to create error.php file.</div>';
    }
}

echo '</div>';

// Check session configuration
echo '<div class="section">
    <h2>6. Checking Session Configuration</h2>';

// Check session directory
$sessionDir = session_save_path();
if (empty($sessionDir)) {
    $sessionDir = sys_get_temp_dir();
}

echo '<div class="info">Session save path: ' . htmlspecialchars($sessionDir) . '</div>';

if (is_writable($sessionDir)) {
    echo '<div class="success">Session directory is writable.</div>';
} else {
    echo '<div class="error">Session directory is not writable. This may cause session-related errors.</div>';
}

// Check session configuration
$sessionMaxLifetime = ini_get('session.gc_maxlifetime');
echo '<div class="info">Session max lifetime: ' . htmlspecialchars($sessionMaxLifetime) . ' seconds</div>';

if (defined('SESSION_LIFETIME') && SESSION_LIFETIME > 0) {
    echo '<div class="success">Custom session lifetime is set to ' . SESSION_LIFETIME . ' seconds.</div>';
} else {
    echo '<div class="warning">Custom session lifetime is not set or is invalid.</div>';
}

echo '</div>';

// Summary and next steps
echo '<div class="section">
    <h2>Summary and Next Steps</h2>';

echo '<p>The diagnostic tool has checked and fixed several potential causes of 500 errors:</p>';
echo '<ul>';
echo '<li>Logs directory and permissions</li>';
echo '<li>Logger functionality</li>';
echo '<li>Database connection</li>';
echo '<li>.htaccess configuration</li>';
echo '<li>Error page</li>';
echo '<li>Session configuration</li>';
echo '</ul>';

echo '<p>If you are still experiencing 500 errors, please try the following:</p>';
echo '<ol>';
echo '<li>Check your server error logs for more specific error messages</li>';
echo '<li>Verify that all required PHP extensions are enabled (PDO, mysqli, mbstring)</li>';
echo '<li>Make sure all files have the correct permissions</li>';
echo '<li>Test your database connection with the correct credentials</li>';
echo '<li>Temporarily enable error display in your PHP configuration</li>';
echo '</ol>';

echo '<p><a href="/new_bot/admin/" class="fix-button">Go to Admin Panel</a></p>';

echo '</div>';

// End HTML output
echo '</body>
</html>';