<?php
/**
 * Database Fix Script
 * 
 * This script fixes database connection issues that cause 500 errors
 */

// Enable error reporting for diagnostic purposes
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set content type to HTML
header('Content-Type: text/html; charset=utf-8');

// Start HTML output
echo '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Database Fix Tool</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            margin: 20px;
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        h1, h2, h3 {
            color: #333;
        }
        .success {
            color: green;
            background-color: #e8f5e9;
            padding: 10px;
            border-left: 5px solid green;
            margin-bottom: 10px;
        }
        .warning {
            color: #856404;
            background-color: #fff3cd;
            padding: 10px;
            border-left: 5px solid #ffeeba;
            margin-bottom: 10px;
        }
        .error {
            color: #721c24;
            background-color: #f8d7da;
            padding: 10px;
            border-left: 5px solid #f5c6cb;
            margin-bottom: 10px;
        }
        .info {
            color: #0c5460;
            background-color: #d1ecf1;
            padding: 10px;
            border-left: 5px solid #bee5eb;
            margin-bottom: 10px;
        }
        pre {
            background-color: #f5f5f5;
            padding: 10px;
            border-radius: 5px;
            overflow-x: auto;
        }
        code {
            font-family: monospace;
            background-color: #f5f5f5;
            padding: 2px 5px;
            border-radius: 3px;
        }
        .section {
            margin-bottom: 30px;
            border: 1px solid #ddd;
            padding: 20px;
            border-radius: 5px;
        }
        .fix-button {
            background-color: #4CAF50;
            color: white;
            padding: 10px 15px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            margin-top: 10px;
        }
        .fix-button:hover {
            background-color: #45a049;
        }
        form {
            margin-top: 20px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        input[type="text"], input[type="password"] {
            width: 100%;
            padding: 8px;
            margin-bottom: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
            box-sizing: border-box;
        }
        .form-group {
            margin-bottom: 15px;
        }
    </style>
</head>
<body>
    <h1>Database Fix Tool</h1>
    <p>This tool diagnoses and fixes database connection issues that cause 500 Internal Server Error in your Telegram Bot Admin Panel.</p>
    
    <div class="section">
        <h2>1. Current Database Configuration</h2>
';

// Include configuration
$configFile = __DIR__ . '/config/config.php';
$configExists = false;
$dbHost = '';
$dbName = '';
$dbUser = '';
$dbPass = '';
$dbCharset = '';

if (file_exists($configFile) && is_readable($configFile)) {
    echo '<div class="success">Config file exists and is readable.</div>';
    $configExists = true;
    
    // Load configuration without requiring it
    $configContent = file_get_contents($configFile);
    
    // Extract database settings using regex
    if (preg_match('/define\s*\(\s*[\'"]DB_HOST[\'"]\s*,\s*[\'"](.*?)[\'"]\s*\)/', $configContent, $matches)) {
        $dbHost = $matches[1];
    }
    
    if (preg_match('/define\s*\(\s*[\'"]DB_NAME[\'"]\s*,\s*[\'"](.*?)[\'"]\s*\)/', $configContent, $matches)) {
        $dbName = $matches[1];
    }
    
    if (preg_match('/define\s*\(\s*[\'"]DB_USER[\'"]\s*,\s*[\'"](.*?)[\'"]\s*\)/', $configContent, $matches)) {
        $dbUser = $matches[1];
    }
    
    if (preg_match('/define\s*\(\s*[\'"]DB_PASS[\'"]\s*,\s*[\'"](.*?)[\'"]\s*\)/', $configContent, $matches)) {
        $dbPass = $matches[1];
    }
    
    if (preg_match('/define\s*\(\s*[\'"]DB_CHARSET[\'"]\s*,\s*[\'"](.*?)[\'"]\s*\)/', $configContent, $matches)) {
        $dbCharset = $matches[1];
    }
    
    echo '<div class="info">Current database settings:</div>';
    echo '<ul>';
    echo '<li>Host: ' . htmlspecialchars($dbHost) . '</li>';
    echo '<li>Database Name: ' . htmlspecialchars($dbName) . '</li>';
    echo '<li>Username: ' . htmlspecialchars($dbUser) . '</li>';
    echo '<li>Password: ' . (empty($dbPass) ? '<span class="warning">Empty</span>' : '<span class="success">Set</span>') . '</li>';
    echo '<li>Charset: ' . htmlspecialchars($dbCharset) . '</li>';
    echo '</ul>';
} else {
    echo '<div class="error">Config file does not exist or is not readable.</div>';
}

echo '</div>';

// Test database connection
echo '<div class="section">
    <h2>2. Testing Database Connection</h2>';

if ($configExists) {
    // Test database connection
    try {
        $dsn = "mysql:host={$dbHost};dbname={$dbName};charset={$dbCharset}";
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ];
        
        $db = new PDO($dsn, $dbUser, $dbPass, $options);
        echo '<div class="success">Database connection successful.</div>';
        
        // Check if database exists
        $stmt = $db->query("SELECT DATABASE()");
        $currentDb = $stmt->fetchColumn();
        
        if ($currentDb == $dbName) {
            echo '<div class="success">Connected to database: ' . htmlspecialchars($currentDb) . '</div>';
        } else {
            echo '<div class="warning">Connected to database ' . htmlspecialchars($currentDb) . ' instead of ' . htmlspecialchars($dbName) . '</div>';
        }
        
        // Check if required tables exist
        $requiredTables = ['users', 'admin_users', 'menus', 'submenus', 'content', 'settings', 'activity_logs'];
        $existingTables = [];
        $missingTables = [];
        
        $stmt = $db->query("SHOW TABLES");
        while ($row = $stmt->fetch(PDO::FETCH_NUM)) {
            $existingTables[] = $row[0];
        }
        
        foreach ($requiredTables as $table) {
            if (!in_array($table, $existingTables)) {
                $missingTables[] = $table;
            }
        }
        
        if (empty($missingTables)) {
            echo '<div class="success">All required tables exist.</div>';
        } else {
            echo '<div class="warning">Missing tables: ' . implode(', ', $missingTables) . '</div>';
            
            // Check if database schema exists
            $schemaFile = __DIR__ . '/database/schema.sql';
            if (file_exists($schemaFile) && is_readable($schemaFile)) {
                echo '<div class="info">Database schema file exists. You can import it to create missing tables.</div>';
                echo '<form method="post" action="">';
                echo '<input type="hidden" name="action" value="import_schema">';
                echo '<button type="submit" class="fix-button">Import Database Schema</button>';
                echo '</form>';
            } else {
                echo '<div class="error">Database schema file does not exist or is not readable.</div>';
            }
        }
        
    } catch (PDOException $e) {
        echo '<div class="error">Database connection failed: ' . htmlspecialchars($e->getMessage()) . '</div>';
        
        // Check if it's an access denied error
        if (strpos($e->getMessage(), 'Access denied') !== false) {
            echo '<div class="info">This appears to be a database access issue. Please check your credentials.</div>';
        }
        
        // Check if it's a database not found error
        if (strpos($e->getMessage(), "Unknown database '{$dbName}'") !== false) {
            echo '<div class="info">The database does not exist. You may need to create it.</div>';
            echo '<form method="post" action="">';
            echo '<input type="hidden" name="action" value="create_database">';
            echo '<button type="submit" class="fix-button">Create Database</button>';
            echo '</form>';
        }
    }
} else {
    echo '<div class="error">Cannot test database connection without configuration.</div>';
}

echo '</div>';

// Update database configuration
echo '<div class="section">
    <h2>3. Update Database Configuration</h2>';

echo '<p>If your database connection is failing, you can update the configuration below:</p>';

echo '<form method="post" action="">';
echo '<input type="hidden" name="action" value="update_config">';

echo '<div class="form-group">';
echo '<label for="db_host">Database Host:</label>';
echo '<input type="text" id="db_host" name="db_host" value="' . htmlspecialchars($dbHost) . '" required>';
echo '</div>';

echo '<div class="form-group">';
echo '<label for="db_name">Database Name:</label>';
echo '<input type="text" id="db_name" name="db_name" value="' . htmlspecialchars($dbName) . '" required>';
echo '</div>';

echo '<div class="form-group">';
echo '<label for="db_user">Database Username:</label>';
echo '<input type="text" id="db_user" name="db_user" value="' . htmlspecialchars($dbUser) . '" required>';
echo '</div>';

echo '<div class="form-group">';
echo '<label for="db_pass">Database Password:</label>';
echo '<input type="password" id="db_pass" name="db_pass" value="' . htmlspecialchars($dbPass) . '">';
echo '</div>';

echo '<div class="form-group">';
echo '<label for="db_charset">Database Charset:</label>';
echo '<input type="text" id="db_charset" name="db_charset" value="' . htmlspecialchars($dbCharset) . '" required>';
echo '</div>';

echo '<button type="submit" class="fix-button">Update Configuration</button>';
echo '</form>';

echo '</div>';

// Process form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        echo '<div class="section">';
        echo '<h2>4. Action Results</h2>';
        
        switch ($_POST['action']) {
            case 'update_config':
                // Update database configuration
                $newDbHost = isset($_POST['db_host']) ? $_POST['db_host'] : $dbHost;
                $newDbName = isset($_POST['db_name']) ? $_POST['db_name'] : $dbName;
                $newDbUser = isset($_POST['db_user']) ? $_POST['db_user'] : $dbUser;
                $newDbPass = isset($_POST['db_pass']) ? $_POST['db_pass'] : $dbPass;
                $newDbCharset = isset($_POST['db_charset']) ? $_POST['db_charset'] : $dbCharset;
                
                if ($configExists && is_writable($configFile)) {
                    $configContent = file_get_contents($configFile);
                    
                    // Update database settings
                    $configContent = preg_replace(
                        '/define\s*\(\s*[\'"]DB_HOST[\'"]\s*,\s*[\'"](.*?)[\'"]\s*\)/',
                        "define('DB_HOST', '{$newDbHost}')",
                        $configContent
                    );
                    
                    $configContent = preg_replace(
                        '/define\s*\(\s*[\'"]DB_NAME[\'"]\s*,\s*[\'"](.*?)[\'"]\s*\)/',
                        "define('DB_NAME', '{$newDbName}')",
                        $configContent
                    );
                    
                    $configContent = preg_replace(
                        '/define\s*\(\s*[\'"]DB_USER[\'"]\s*,\s*[\'"](.*?)[\'"]\s*\)/',
                        "define('DB_USER', '{$newDbUser}')",
                        $configContent
                    );
                    
                    $configContent = preg_replace(
                        '/define\s*\(\s*[\'"]DB_PASS[\'"]\s*,\s*[\'"](.*?)[\'"]\s*\)/',
                        "define('DB_PASS', '{$newDbPass}')",
                        $configContent
                    );
                    
                    $configContent = preg_replace(
                        '/define\s*\(\s*[\'"]DB_CHARSET[\'"]\s*,\s*[\'"](.*?)[\'"]\s*\)/',
                        "define('DB_CHARSET', '{$newDbCharset}')",
                        $configContent
                    );
                    
                    if (file_put_contents($configFile, $configContent)) {
                        echo '<div class="success">Database configuration updated successfully.</div>';
                        echo '<div class="info">Please refresh this page to test the new configuration.</div>';
                    } else {
                        echo '<div class="error">Failed to update database configuration.</div>';
                    }
                } else {
                    echo '<div class="error">Config file does not exist or is not writable.</div>';
                }
                break;
                
            case 'create_database':
                // Create database
                try {
                    $dsn = "mysql:host={$dbHost};charset={$dbCharset}";
                    $options = [
                        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                        PDO::ATTR_EMULATE_PREPARES => false,
                    ];
                    
                    $db = new PDO($dsn, $dbUser, $dbPass, $options);
                    
                    // Create database
                    $stmt = $db->prepare("CREATE DATABASE IF NOT EXISTS `{$dbName}`");
                    if ($stmt->execute()) {
                        echo '<div class="success">Database created successfully.</div>';
                        echo '<div class="info">Please refresh this page to test the connection.</div>';
                    } else {
                        echo '<div class="error">Failed to create database.</div>';
                    }
                } catch (PDOException $e) {
                    echo '<div class="error">Failed to create database: ' . htmlspecialchars($e->getMessage()) . '</div>';
                }
                break;
                
            case 'import_schema':
                // Import database schema
                $schemaFile = __DIR__ . '/database/schema.sql';
                if (file_exists($schemaFile) && is_readable($schemaFile)) {
                    try {
                        $dsn = "mysql:host={$dbHost};dbname={$dbName};charset={$dbCharset}";
                        $options = [
                            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                            PDO::ATTR_EMULATE_PREPARES => false,
                        ];
                        
                        $db = new PDO($dsn, $dbUser, $dbPass, $options);
                        
                        // Read schema file
                        $schema = file_get_contents($schemaFile);
                        
                        // Split into individual queries
                        $queries = explode(';', $schema);
                        
                        // Execute each query
                        $successCount = 0;
                        $errorCount = 0;
                        
                        foreach ($queries as $query) {
                            $query = trim($query);
                            if (!empty($query)) {
                                try {
                                    $db->exec($query);
                                    $successCount++;
                                } catch (PDOException $e) {
                                    $errorCount++;
                                    echo '<div class="error">Error executing query: ' . htmlspecialchars($e->getMessage()) . '</div>';
                                }
                            }
                        }
                        
                        echo '<div class="success">Schema import completed. ' . $successCount . ' queries executed successfully, ' . $errorCount . ' errors.</div>';
                        echo '<div class="info">Please refresh this page to verify tables.</div>';
                    } catch (PDOException $e) {
                        echo '<div class="error">Failed to connect to database: ' . htmlspecialchars($e->getMessage()) . '</div>';
                    }
                } else {
                    echo '<div class="error">Schema file does not exist or is not readable.</div>';
                }
                break;
        }
        
        echo '</div>';
    }
}

// Summary and next steps
echo '<div class="section">
    <h2>Summary and Next Steps</h2>';

echo '<p>This tool has checked your database configuration and connection status.</p>';

echo '<p>If you are still experiencing 500 errors after fixing database issues, please try the following:</p>';
echo '<ol>';
echo '<li>Run the <a href="fix_500_errors.php">500 Error Fix Tool</a> to check for other issues</li>';
echo '<li>Check your server error logs for more specific error messages</li>';
echo '<li>Verify that all required PHP extensions are enabled (PDO, mysqli, mbstring)</li>';
echo '<li>Make sure all files have the correct permissions</li>';
echo '</ol>';

echo '<p><a href="/new_bot/admin/" class="fix-button">Go to Admin Panel</a></p>';

echo '</div>';

// End HTML output
echo '</body>
</html>';