<?php
/**
 * Database Access Troubleshooter
 * 
 * This script helps diagnose and fix database access issues
 * including the "Access denied" error for MySQL users.
 */

// Enable error reporting for diagnostic purposes
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set content type to HTML
header('Content-Type: text/html; charset=utf-8');

// Start HTML output
echo '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Database Access Troubleshooter</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            margin: 20px;
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        h1, h2, h3 {
            color: #333;
        }
        .success {
            color: green;
            background-color: #e8f5e9;
            padding: 10px;
            border-left: 5px solid green;
            margin-bottom: 10px;
        }
        .warning {
            color: #856404;
            background-color: #fff3cd;
            padding: 10px;
            border-left: 5px solid #ffeeba;
            margin-bottom: 10px;
        }
        .error {
            color: #721c24;
            background-color: #f8d7da;
            padding: 10px;
            border-left: 5px solid #f5c6cb;
            margin-bottom: 10px;
        }
        .info {
            color: #0c5460;
            background-color: #d1ecf1;
            padding: 10px;
            border-left: 5px solid #bee5eb;
            margin-bottom: 10px;
        }
        pre {
            background-color: #f5f5f5;
            padding: 10px;
            border-radius: 5px;
            overflow-x: auto;
        }
        code {
            font-family: monospace;
            background-color: #f5f5f5;
            padding: 2px 5px;
            border-radius: 3px;
        }
        .section {
            margin-bottom: 30px;
            border: 1px solid #ddd;
            padding: 20px;
            border-radius: 5px;
        }
        table {
            border-collapse: collapse;
            width: 100%;
            margin-bottom: 20px;
        }
        th, td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: left;
        }
        th {
            background-color: #f2f2f2;
        }
        .fix-button {
            background-color: #4CAF50;
            color: white;
            padding: 10px 15px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 16px;
            margin-top: 10px;
        }
        .fix-button:hover {
            background-color: #45a049;
        }
        form {
            margin-bottom: 20px;
        }
        input[type=text], input[type=password] {
            width: 100%;
            padding: 12px 20px;
            margin: 8px 0;
            display: inline-block;
            border: 1px solid #ccc;
            border-radius: 4px;
            box-sizing: border-box;
        }
        .form-group {
            margin-bottom: 15px;
        }
        label {
            font-weight: bold;
        }
    </style>
</head>
<body>
    <h1>Database Access Troubleshooter</h1>
    <p>This tool helps diagnose and fix database access issues, including "Access denied" errors.</p>
';

// Initialize variables
$issues = [];
$fixes = [];
$errors = [];
$warnings = [];
$success = [];

// Function to check if config file exists and get database settings
function getConfigSettings() {
    $configFile = dirname(__DIR__) . '/config/config.php';
    
    if (!file_exists($configFile)) {
        echo "<div class='error'>❌ Configuration file not found: {$configFile}</div>";
        return false;
    }
    
    // Include configuration file
    include_once $configFile;
    
    // Check if database constants are defined
    if (!defined('DB_HOST') || !defined('DB_NAME') || !defined('DB_USER') || !defined('DB_PASS')) {
        echo "<div class='error'>❌ Database configuration constants not defined in config file</div>";
        return false;
    }
    
    return [
        'host' => DB_HOST,
        'name' => DB_NAME,
        'user' => DB_USER,
        'pass' => DB_PASS,
        'charset' => defined('DB_CHARSET') ? DB_CHARSET : 'utf8mb4'
    ];
}

// Function to test database connection
function testDatabaseConnection($host, $user, $pass, $name = null) {
    try {
        $dsn = "mysql:host={$host}" . ($name ? ";dbname={$name}" : '');
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ];
        
        $db = new PDO($dsn, $user, $pass, $options);
        return ['success' => true, 'message' => 'Connection successful'];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Function to check if user has database creation privileges
function checkUserPrivileges($host, $user, $pass) {
    try {
        $dsn = "mysql:host={$host}";
        $db = new PDO($dsn, $user, $pass);
        
        $stmt = $db->query("SHOW GRANTS FOR CURRENT_USER()");
        $grants = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        $hasCreateDb = false;
        $hasAllPrivileges = false;
        
        foreach ($grants as $grant) {
            if (strpos($grant, 'ALL PRIVILEGES ON *.*') !== false) {
                $hasAllPrivileges = true;
            }
            if (strpos($grant, 'CREATE') !== false) {
                $hasCreateDb = true;
            }
        }
        
        return [
            'success' => true,
            'has_create_db' => $hasCreateDb || $hasAllPrivileges,
            'has_all_privileges' => $hasAllPrivileges,
            'grants' => $grants
        ];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Function to create database
function createDatabase($host, $user, $pass, $name) {
    try {
        $dsn = "mysql:host={$host}";
        $db = new PDO($dsn, $user, $pass);
        
        // Create database if it doesn't exist
        $db->exec("CREATE DATABASE IF NOT EXISTS `{$name}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        
        return ['success' => true, 'message' => "Database '{$name}' created successfully"];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Function to grant privileges to user
function grantPrivileges($host, $adminUser, $adminPass, $dbName, $dbUser) {
    try {
        $dsn = "mysql:host={$host}";
        $db = new PDO($dsn, $adminUser, $adminPass);
        
        // Grant privileges to user
        $db->exec("GRANT ALL PRIVILEGES ON `{$dbName}`.* TO '{$dbUser}'@'localhost'");
        $db->exec("FLUSH PRIVILEGES");
        
        return ['success' => true, 'message' => "Privileges granted to user '{$dbUser}' on database '{$dbName}'"];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Function to check if database exists
function checkDatabaseExists($host, $user, $pass, $name) {
    try {
        $dsn = "mysql:host={$host}";
        $db = new PDO($dsn, $user, $pass);
        
        $stmt = $db->query("SELECT SCHEMA_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = '{$name}'");
        $result = $stmt->fetchColumn();
        
        return ['success' => true, 'exists' => !empty($result)];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Main execution
echo "<div class='section'><h2>Database Configuration</h2>";

// Get configuration settings
$config = getConfigSettings();

if ($config) {
    echo "<div class='info'>ℹ️ Database configuration found:</div>";
    echo "<ul>";
    echo "<li><strong>Host:</strong> {$config['host']}</li>";
    echo "<li><strong>Database:</strong> {$config['name']}</li>";
    echo "<li><strong>User:</strong> {$config['user']}</li>";
    echo "<li><strong>Password:</strong> " . str_repeat('*', strlen($config['pass'])) . "</li>";
    echo "<li><strong>Charset:</strong> {$config['charset']}</li>";
    echo "</ul>";
    
    // Test connection to MySQL server (without database)
    echo "<h3>Testing MySQL Server Connection</h3>";
    $serverConnection = testDatabaseConnection($config['host'], $config['user'], $config['pass']);
    
    if ($serverConnection['success']) {
        echo "<div class='success'>✅ Successfully connected to MySQL server</div>";
        
        // Check user privileges
        $privileges = checkUserPrivileges($config['host'], $config['user'], $config['pass']);
        
        if ($privileges['success']) {
            echo "<h3>User Privileges</h3>";
            
            if ($privileges['has_all_privileges']) {
                echo "<div class='success'>✅ User has ALL PRIVILEGES</div>";
            } elseif ($privileges['has_create_db']) {
                echo "<div class='success'>✅ User has CREATE DATABASE privilege</div>";
            } else {
                echo "<div class='warning'>⚠️ User does not have CREATE DATABASE privilege</div>";
                $warnings[] = "User does not have CREATE DATABASE privilege";
            }
            
            echo "<h4>Grants:</h4>";
            echo "<ul>";
            foreach ($privileges['grants'] as $grant) {
                echo "<li>{$grant}</li>";
            }
            echo "</ul>";
            
            // Check if database exists
            $dbExists = checkDatabaseExists($config['host'], $config['user'], $config['pass'], $config['name']);
            
            if ($dbExists['success']) {
                if ($dbExists['exists']) {
                    echo "<div class='success'>✅ Database '{$config['name']}' exists</div>";
                    
                    // Test connection to the database
                    $dbConnection = testDatabaseConnection($config['host'], $config['user'], $config['pass'], $config['name']);
                    
                    if ($dbConnection['success']) {
                        echo "<div class='success'>✅ Successfully connected to database '{$config['name']}'</div>";
                        $success[] = "Database connection is working properly";
                    } else {
                        echo "<div class='error'>❌ Failed to connect to database '{$config['name']}': {$dbConnection['message']}</div>";
                        $issues[] = "Cannot connect to existing database: {$dbConnection['message']}";
                        
                        // Check if it's an access denied error
                        if (strpos($dbConnection['message'], 'Access denied') !== false) {
                            echo "<div class='info'>ℹ️ This is an access denied error. The database exists but the user does not have access to it.</div>";
                            $fixes[] = "Grant privileges to user '{$config['user']}' on database '{$config['name']}'";
                        }
                    }
                } else {
                    echo "<div class='error'>❌ Database '{$config['name']}' does not exist</div>";
                    $issues[] = "Database '{$config['name']}' does not exist";
                    
                    if ($privileges['has_create_db']) {
                        echo "<div class='info'>ℹ️ User has permission to create the database</div>";
                        $fixes[] = "Create database '{$config['name']}'";
                    } else {
                        echo "<div class='warning'>⚠️ User does not have permission to create the database</div>";
                        $fixes[] = "Use a MySQL admin account to create the database and grant privileges";
                    }
                }
            } else {
                echo "<div class='error'>❌ Failed to check if database exists: {$dbExists['message']}</div>";
                $issues[] = "Failed to check if database exists: {$dbExists['message']}";
            }
        } else {
            echo "<div class='error'>❌ Failed to check user privileges: {$privileges['message']}</div>";
            $issues[] = "Failed to check user privileges: {$privileges['message']}";
        }
    } else {
        echo "<div class='error'>❌ Failed to connect to MySQL server: {$serverConnection['message']}</div>";
        $issues[] = "Cannot connect to MySQL server: {$serverConnection['message']}";
        
        // Check if it's an access denied error
        if (strpos($serverConnection['message'], 'Access denied') !== false) {
            echo "<div class='info'>ℹ️ This is an access denied error. The MySQL user credentials may be incorrect.</div>";
            $fixes[] = "Verify MySQL user credentials";
        }
    }
}

echo "</div>";

// Display fix form if there are issues
if (!empty($issues)) {
    echo "<div class='section'><h2>Fix Database Issues</h2>";
    
    // Create database form
    if (in_array("Database '{$config['name']}' does not exist", $issues)) {
        echo "<h3>Create Database</h3>";
        echo "<form method='post' action=''>";
        echo "<input type='hidden' name='action' value='create_database'>";
        
        if (!$privileges['has_create_db']) {
            echo "<div class='info'>ℹ️ Your current MySQL user does not have permission to create databases. Please enter MySQL admin credentials:</div>";
            echo "<div class='form-group'>";
            echo "<label for='admin_user'>MySQL Admin Username:</label>";
            echo "<input type='text' id='admin_user' name='admin_user' required>";
            echo "</div>";
            echo "<div class='form-group'>";
            echo "<label for='admin_pass'>MySQL Admin Password:</label>";
            echo "<input type='password' id='admin_pass' name='admin_pass' required>";
            echo "</div>";
        }
        
        echo "<button type='submit' class='fix-button'>Create Database</button>";
        echo "</form>";
    }
    
    // Grant privileges form
    if (in_array("Cannot connect to existing database: {$dbConnection['message']}", $issues) && 
        strpos($dbConnection['message'], 'Access denied') !== false) {
        echo "<h3>Grant Privileges</h3>";
        echo "<form method='post' action=''>";
        echo "<input type='hidden' name='action' value='grant_privileges'>";
        echo "<div class='info'>ℹ️ Your current MySQL user does not have access to the database. Please enter MySQL admin credentials to grant privileges:</div>";
        echo "<div class='form-group'>";
        echo "<label for='admin_user'>MySQL Admin Username:</label>";
        echo "<input type='text' id='admin_user' name='admin_user' required>";
        echo "</div>";
        echo "<div class='form-group'>";
        echo "<label for='admin_pass'>MySQL Admin Password:</label>";
        echo "<input type='password' id='admin_pass' name='admin_pass' required>";
        echo "</div>";
        echo "<button type='submit' class='fix-button'>Grant Privileges</button>";
        echo "</form>";
    }
    
    // Update configuration form
    if (in_array("Cannot connect to MySQL server: {$serverConnection['message']}", $issues) && 
        strpos($serverConnection['message'], 'Access denied') !== false) {
        echo "<h3>Update Database Configuration</h3>";
        echo "<form method='post' action=''>";
        echo "<input type='hidden' name='action' value='update_config'>";
        echo "<div class='info'>ℹ️ The MySQL credentials in your configuration file may be incorrect. Please enter the correct credentials:</div>";
        echo "<div class='form-group'>";
        echo "<label for='db_host'>Database Host:</label>";
        echo "<input type='text' id='db_host' name='db_host' value='{$config['host']}' required>";
        echo "</div>";
        echo "<div class='form-group'>";
        echo "<label for='db_name'>Database Name:</label>";
        echo "<input type='text' id='db_name' name='db_name' value='{$config['name']}' required>";
        echo "</div>";
        echo "<div class='form-group'>";
        echo "<label for='db_user'>Database Username:</label>";
        echo "<input type='text' id='db_user' name='db_user' value='{$config['user']}' required>";
        echo "</div>";
        echo "<div class='form-group'>";
        echo "<label for='db_pass'>Database Password:</label>";
        echo "<input type='password' id='db_pass' name='db_pass' placeholder='Enter new password' required>";
        echo "</div>";
        echo "<button type='submit' class='fix-button'>Update Configuration</button>";
        echo "</form>";
    }
    
    echo "</div>";
}

// Process form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    echo "<div class='section'><h2>Action Results</h2>";
    
    switch ($action) {
        case 'create_database':
            if (isset($_POST['admin_user']) && isset($_POST['admin_pass'])) {
                // Create database using admin credentials
                $result = createDatabase($config['host'], $_POST['admin_user'], $_POST['admin_pass'], $config['name']);
                
                if ($result['success']) {
                    echo "<div class='success'>✅ {$result['message']}</div>";
                    
                    // Grant privileges to the configured user
                    $grantResult = grantPrivileges($config['host'], $_POST['admin_user'], $_POST['admin_pass'], $config['name'], $config['user']);
                    
                    if ($grantResult['success']) {
                        echo "<div class='success'>✅ {$grantResult['message']}</div>";
                    } else {
                        echo "<div class='error'>❌ Failed to grant privileges: {$grantResult['message']}</div>";
                    }
                } else {
                    echo "<div class='error'>❌ Failed to create database: {$result['message']}</div>";
                }
            } else {
                // Create database using configured user
                $result = createDatabase($config['host'], $config['user'], $config['pass'], $config['name']);
                
                if ($result['success']) {
                    echo "<div class='success'>✅ {$result['message']}</div>";
                } else {
                    echo "<div class='error'>❌ Failed to create database: {$result['message']}</div>";
                }
            }
            break;
            
        case 'grant_privileges':
            if (isset($_POST['admin_user']) && isset($_POST['admin_pass'])) {
                $result = grantPrivileges($config['host'], $_POST['admin_user'], $_POST['admin_pass'], $config['name'], $config['user']);
                
                if ($result['success']) {
                    echo "<div class='success'>✅ {$result['message']}</div>";
                } else {
                    echo "<div class='error'>❌ Failed to grant privileges: {$result['message']}</div>";
                }
            } else {
                echo "<div class='error'>❌ MySQL admin credentials are required</div>";
            }
            break;
            
        case 'update_config':
            if (isset($_POST['db_host']) && isset($_POST['db_name']) && isset($_POST['db_user']) && isset($_POST['db_pass'])) {
                // Test connection with new credentials
                $testResult = testDatabaseConnection($_POST['db_host'], $_POST['db_user'], $_POST['db_pass']);
                
                if ($testResult['success']) {
                    echo "<div class='success'>✅ Connection with new credentials successful</div>";
                    
                    // Update config file
                    $configFile = dirname(__DIR__) . '/config/config.php';
                    $configContent = file_get_contents($configFile);
                    
                    // Replace database settings
                    $configContent = preg_replace("/define\('DB_HOST', '.*?'\);/", "define('DB_HOST', '{$_POST['db_host']}');", $configContent);
                    $configContent = preg_replace("/define\('DB_NAME', '.*?'\);/", "define('DB_NAME', '{$_POST['db_name']}');", $configContent);
                    $configContent = preg_replace("/define\('DB_USER', '.*?'\);/", "define('DB_USER', '{$_POST['db_user']}');", $configContent);
                    $configContent = preg_replace("/define\('DB_PASS', '.*?'\);/", "define('DB_PASS', '{$_POST['db_pass']}');", $configContent);
                    
                    if (file_put_contents($configFile, $configContent)) {
                        echo "<div class='success'>✅ Configuration file updated successfully</div>";
                    } else {
                        echo "<div class='error'>❌ Failed to update configuration file</div>";
                    }
                } else {
                    echo "<div class='error'>❌ Connection with new credentials failed: {$testResult['message']}</div>";
                }
            } else {
                echo "<div class='error'>❌ All database configuration fields are required</div>";
            }
            break;
    }
    
    echo "<p><a href='?'>Run diagnostics again</a></p>";
    echo "</div>";
}

// Summary
echo "<div class='section'><h2>Summary</h2>";

if (!empty($success)) {
    echo "<h3>Successful Checks:</h3>";
    echo "<ul>";
    foreach ($success as $item) {
        echo "<li>{$item}</li>";
    }
    echo "</ul>";
}

if (!empty($warnings)) {
    echo "<h3>Warnings:</h3>";
    echo "<ul>";
    foreach ($warnings as $warning) {
        echo "<li>{$warning}</li>";
    }
    echo "</ul>";
}

if (!empty($issues)) {
    echo "<h3>Issues:</h3>";
    echo "<ul>";
    foreach ($issues as $issue) {
        echo "<li>{$issue}</li>";
    }
    echo "</ul>";
}

if (!empty($fixes)) {
    echo "<h3>Recommended Fixes:</h3>";
    echo "<ul>";
    foreach ($fixes as $fix) {
        echo "<li>{$fix}</li>";
    }
    echo "</ul>";
}

echo "</div>";

// Database creation instructions
echo "<div class='section'><h2>Manual Database Setup Instructions</h2>";

echo "<h3>Option 1: Using phpMyAdmin</h3>";
echo "<ol>";
echo "<li>Log in to phpMyAdmin with your hosting control panel</li>";
echo "<li>Click on 'Databases' tab</li>";
echo "<li>Enter '{$config['name']}' in the 'Create database' field</li>";
echo "<li>Select 'utf8mb4_unicode_ci' from the collation dropdown</li>";
echo "<li>Click 'Create' button</li>";
echo "<li>Go to 'Users' tab</li>";
echo "<li>Click 'Add user' or edit existing user '{$config['user']}'</li>";
echo "<li>Grant all privileges on the database '{$config['name']}' to user '{$config['user']}'</li>";
echo "</ol>";

echo "<h3>Option 2: Using MySQL Command Line</h3>";
echo "<pre>";
echo "# Login as MySQL admin\nmysql -u root -p\n\n# Create database\nCREATE DATABASE IF NOT EXISTS `{$config['name']}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;\n\n# Create user if it doesn't exist\nCREATE USER IF NOT EXISTS '{$config['user']}'@'localhost' IDENTIFIED BY 'your_password';\n\n# Grant privileges\nGRANT ALL PRIVILEGES ON `{$config['name']}`.* TO '{$config['user']}'@'localhost';\n\n# Apply changes\nFLUSH PRIVILEGES;\n";
echo "</pre>";

echo "<h3>Option 3: Contact Your Hosting Provider</h3>";
echo "<p>If you don't have access to phpMyAdmin or MySQL command line, contact your hosting provider with the following information:</p>";
echo "<ul>";
echo "<li>Request to create a new database named '{$config['name']}'</li>";
echo "<li>Request to grant all privileges on this database to user '{$config['user']}'</li>";
echo "<li>Ensure the database uses UTF-8 (utf8mb4) character encoding</li>";
echo "</ul>";

echo "</div>";

// End HTML output
echo '</body>
</html>';