<?php
/**
 * Fix Schema Database Name
 * 
 * This script modifies the schema.sql file to use the correct database name
 * from config.php to resolve the access denied error.
 */

// Load configuration
require_once __DIR__ . '/config/config.php';

// Define paths
$schemaFile = __DIR__ . '/database/schema.sql';
$backupFile = __DIR__ . '/database/schema.sql.backup.' . date('Y-m-d-His');

// Display header
echo "===================================\n";
echo "Database Schema Fixer\n";
echo "===================================\n\n";

// Check if schema file exists
if (!file_exists($schemaFile)) {
    die("Error: Schema file not found at {$schemaFile}\n");
}

// Create backup
if (!copy($schemaFile, $backupFile)) {
    die("Error: Failed to create backup of schema file\n");
}
echo "✓ Created backup at {$backupFile}\n";

// Read schema file
$schemaContent = file_get_contents($schemaFile);
if ($schemaContent === false) {
    die("Error: Failed to read schema file\n");
}

// Get database name from config
$configDbName = defined('DB_NAME') ? DB_NAME : 'mbofzzfa_new_bot_db';
echo "✓ Found database name in config: {$configDbName}\n";

// Find current database name in schema
preg_match('/CREATE DATABASE IF NOT EXISTS ([^;]+);/', $schemaContent, $matches);
if (empty($matches[1])) {
    die("Error: Could not find database creation statement in schema file\n");
}
$schemaDbName = trim($matches[1]);
echo "✓ Found database name in schema: {$schemaDbName}\n";

// Check if they're already the same
if ($schemaDbName === $configDbName) {
    echo "\nDatabase names already match. No changes needed.\n";
    exit(0);
}

// Replace database name in schema
$updatedSchema = str_replace(
    "CREATE DATABASE IF NOT EXISTS {$schemaDbName};",
    "CREATE DATABASE IF NOT EXISTS {$configDbName};",
    $schemaContent
);

$updatedSchema = str_replace(
    "USE {$schemaDbName};",
    "USE {$configDbName};",
    $updatedSchema
);

// Write updated schema
if (file_put_contents($schemaFile, $updatedSchema) === false) {
    die("Error: Failed to write updated schema file\n");
}

echo "\n✅ Successfully updated schema file to use database name '{$configDbName}'\n";
echo "\nYou can now run the schema using:\n";
echo "- phpMyAdmin: Import the schema.sql file\n";
echo "- Command line: mysql -u " . DB_USER . " -p < {$schemaFile}\n";
echo "\nIf you still encounter permission issues, you may need to:\n";
echo "1. Create the database manually first: CREATE DATABASE {$configDbName};\n";
echo "2. Grant permissions: GRANT ALL PRIVILEGES ON {$configDbName}.* TO '" . DB_USER . "'@'localhost';\n";
echo "3. Flush privileges: FLUSH PRIVILEGES;\n";

echo "\nDone!\n";