<?php
/**
 * Logger Class
 * Handles logging for the Telegram Bot
 */

class Logger {
    private $logDir;
    private $logFile;
    private $errorLogFile;
    
    /**
     * Constructor
     */
    public function __construct() {
        // Set log directory
        $this->logDir = dirname(__DIR__) . '/logs';
        
        // Create log directory if it doesn't exist
        if (!file_exists($this->logDir)) {
            mkdir($this->logDir, 0755, true);
        }
        
        // Set log files
        $this->logFile = $this->logDir . '/bot.log';
        $this->errorLogFile = $this->logDir . '/error.log';
    }
    
    /**
     * Log a message
     * @param string $message Message to log
     * @param string $level Log level (info, warning, error)
     * @return bool Success status
     */
    public function log($message, $level = 'info') {
        // Format log entry
        $logEntry = date('[Y-m-d H:i:s]') . ' [' . strtoupper($level) . '] ' . $message . PHP_EOL;
        
        // Write to appropriate log file
        if ($level == 'error') {
            return file_put_contents($this->errorLogFile, $logEntry, FILE_APPEND);
        } else {
            return file_put_contents($this->logFile, $logEntry, FILE_APPEND);
        }
    }
    
    /**
     * Log an error message
     * @param string $message Error message
     * @return bool Success status
     */
    public function error($message) {
        return $this->log($message, 'error');
    }
    
    /**
     * Log a warning message
     * @param string $message Warning message
     * @return bool Success status
     */
    public function warning($message) {
        return $this->log($message, 'warning');
    }
    
    /**
     * Log an info message
     * @param string $message Info message
     * @return bool Success status
     */
    public function info($message) {
        return $this->log($message, 'info');
    }
    
    /**
     * Log a debug message (only if debug mode is enabled)
     * @param string $message Debug message
     * @return bool Success status or null if debug mode is disabled
     */
    public function debug($message) {
        if (defined('DEBUG_MODE') && DEBUG_MODE) {
            return $this->log($message, 'debug');
        }
        return null;
    }
    
    /**
     * Get log file contents
     * @param string $type Log type (bot, error)
     * @param int $lines Number of lines to return (0 for all)
     * @return string Log contents
     */
    public function getLogContents($type = 'bot', $lines = 100) {
        $file = ($type == 'error') ? $this->errorLogFile : $this->logFile;
        
        if (!file_exists($file)) {
            return '';
        }
        
        if ($lines <= 0) {
            return file_get_contents($file);
        } else {
            // Get last N lines
            $logContents = file($file);
            $logContents = array_slice($logContents, -$lines);
            return implode('', $logContents);
        }
    }
    
    /**
     * Clear log file
     * @param string $type Log type (bot, error, all)
     * @return bool Success status
     */
    public function clearLog($type = 'all') {
        $success = true;
        
        if ($type == 'all' || $type == 'bot') {
            $success = $success && file_put_contents($this->logFile, '');
        }
        
        if ($type == 'all' || $type == 'error') {
            $success = $success && file_put_contents($this->errorLogFile, '');
        }
        
        return $success;
    }
}