<?php
/**
 * Helper functions for Telegram Bot
 */

/**
 * Get setting value from database
 * @param PDO $db Database connection
 * @param string $key Setting key
 * @param mixed $default Default value if setting not found
 * @return mixed Setting value
 */
function getSetting($db, $key, $default = null) {
    $stmt = $db->prepare("SELECT value FROM settings WHERE setting_key = ?");
    $stmt->execute([$key]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    return $result ? json_decode($result['value'], true) : $default;
}

/**
 * Update setting in database
 * @param PDO $db Database connection
 * @param string $key Setting key
 * @param mixed $value Setting value
 * @return bool Success status
 */
function updateSetting($db, $key, $value) {
    // Encode value as JSON
    $jsonValue = json_encode($value);
    
    // Check if setting exists
    $stmt = $db->prepare("SELECT COUNT(*) FROM settings WHERE setting_key = ?");
    $stmt->execute([$key]);
    $exists = (bool)$stmt->fetchColumn();
    
    if ($exists) {
        // Update existing setting
        $stmt = $db->prepare("UPDATE settings SET value = ? WHERE setting_key = ?");
        return $stmt->execute([$jsonValue, $key]);
    } else {
        // Insert new setting
        $stmt = $db->prepare("INSERT INTO settings (setting_key, value) VALUES (?, ?)");
        return $stmt->execute([$key, $jsonValue]);
    }
}

// Deprecated: logActivity is defined centrally in config/config.php with extended signature. Removed here to avoid function redeclaration and schema mismatch.

/**
 * Format message text for Telegram Markdown
 * @param string $text Text to format
 * @return string Formatted text
 */
function formatMessageText($text) {
    // Escape special characters
    $text = str_replace(['_', '*', '`', '['], ['\\_', '\\*', '\\`', '\\['], $text);
    return $text;
}

/**
 * Generate inline keyboard markup
 * @param array $buttons Array of buttons
 * @return array Inline keyboard markup
 */
function inlineKeyboard($buttons) {
    return [
        'inline_keyboard' => $buttons
    ];
}

/**
 * Get menu by ID
 * @param PDO $db Database connection
 * @param int $menuId Menu ID
 * @return array|bool Menu data or false if not found
 */
function getMenu($db, $menuId) {
    $stmt = $db->prepare("SELECT * FROM menus WHERE id = ?");
    $stmt->execute([$menuId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Get submenu by ID
 * @param PDO $db Database connection
 * @param int $submenuId Submenu ID
 * @return array|bool Submenu data or false if not found
 */
function getSubmenu($db, $submenuId) {
    $stmt = $db->prepare("SELECT * FROM submenus WHERE id = ?");
    $stmt->execute([$submenuId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Get content by ID
 * @param PDO $db Database connection
 * @param int $contentId Content ID
 * @return array|bool Content data or false if not found
 */
function getContent($db, $contentId) {
    $stmt = $db->prepare("SELECT * FROM content WHERE id = ?");
    $stmt->execute([$contentId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Get links by submenu ID
 * @param PDO $db Database connection
 * @param int $submenuId Submenu ID
 * @return array Links data
 */
function getLinksBySubmenu($db, $submenuId) {
    $stmt = $db->prepare("SELECT * FROM links WHERE submenu_id = ? ORDER BY title");
    $stmt->execute([$submenuId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Deprecated: isAdmin is now defined in config/config.php. Removed here to avoid function redeclaration.

/**
 * Generate a random string
 * @param int $length Length of the string
 * @return string Random string
 */
function generateRandomString($length = 10) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $charactersLength = strlen($characters);
    $randomString = '';
    for ($i = 0; $i < $length; $i++) {
        $randomString .= $characters[rand(0, $charactersLength - 1)];
    }
    return $randomString;
}

/**
 * Process callback query data
 * @param TelegramBot $bot Bot instance
 * @param int $chatId Chat ID
 * @param int $messageId Message ID
 * @param string $data Callback data
 * @param string $callbackId Callback query ID
 * @param array $user User data
 */
function processCallbackQuery($bot, $chatId, $messageId, $data, $callbackId, $user) {
    global $db;
    
    // Answer callback query to remove loading state
    $bot->answerCallbackQuery($callbackId);
    
    // Parse callback data
    $parts = explode(':', $data);
    $action = $parts[0];
    
    switch ($action) {
        case 'menu':
            // Show main menu
            if (count($parts) > 1) {
                $menuId = (int)$parts[1];
                showMenu($bot, $chatId, $messageId, $menuId);
            } else {
                showMainMenu($bot, $chatId, $messageId);
            }
            break;
            
        case 'submenu':
            // Show submenu
            if (count($parts) > 1) {
                $submenuId = (int)$parts[1];
                showSubmenu($bot, $chatId, $messageId, $submenuId);
            }
            break;
            
        case 'content':
            // Show content
            if (count($parts) > 1) {
                $contentId = (int)$parts[1];
                showContent($bot, $chatId, $messageId, $contentId);
            }
            break;
            
        case 'back':
            // Go back to previous menu
            if (count($parts) > 1) {
                if ($parts[1] == 'main') {
                    showMainMenu($bot, $chatId, $messageId);
                } else {
                    $menuId = (int)$parts[1];
                    showMenu($bot, $chatId, $messageId, $menuId);
                }
            } else {
                showMainMenu($bot, $chatId, $messageId);
            }
            break;
    }
    
    // Log activity
    logActivity('user', $user['id'], null, 'callback_query', $data, $_SERVER['REMOTE_ADDR'] ?? '');
}

/**
 * Show main menu
 * @param TelegramBot $bot Bot instance
 * @param int $chatId Chat ID
 * @param int|null $messageId Message ID (null for new message)
 */
function showMainMenu($bot, $chatId, $messageId = null) {
    global $db;
    
    // Get all main menus
    $stmt = $db->prepare("SELECT * FROM menus WHERE parent_id IS NULL ORDER BY position");
    $stmt->execute();
    $menus = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Create buttons
    $buttons = [];
    foreach ($menus as $menu) {
        $buttons[] = [['text' => $menu['title'], 'callback_data' => 'menu:' . $menu['id']]];
    }
    
    // Get welcome message from settings
    $welcomeMessage = getSetting($db, 'welcome_message', MESSAGES['welcome']);
    
    // Send or edit message
    $options = [
        'reply_markup' => json_encode(inlineKeyboard($buttons)),
        'parse_mode' => 'Markdown'
    ];
    
    if ($messageId) {
        $bot->editMessageText($chatId, $messageId, $welcomeMessage, $options);
    } else {
        $bot->sendMessage($chatId, $welcomeMessage, $options);
    }
}

/**
 * Show menu
 * @param TelegramBot $bot Bot instance
 * @param int $chatId Chat ID
 * @param int $messageId Message ID
 * @param int $menuId Menu ID
 */
function showMenu($bot, $chatId, $messageId, $menuId) {
    global $db;
    
    // Get menu
    $menu = getMenu($db, $menuId);
    if (!$menu) {
        $bot->editMessageText($chatId, $messageId, MESSAGES['not_found']);
        return;
    }
    
    // Get submenus for this menu
    $stmt = $db->prepare("SELECT * FROM submenus WHERE menu_id = ? ORDER BY position");
    $stmt->execute([$menuId]);
    $submenus = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Create buttons
    $buttons = [];
    foreach ($submenus as $submenu) {
        $buttons[] = [['text' => $submenu['title'], 'callback_data' => 'submenu:' . $submenu['id']]];
    }
    
    // Add back button
    $buttons[] = [['text' => '« Back', 'callback_data' => 'back:main']];
    
    // Edit message
    $bot->editMessageText($chatId, $messageId, $menu['title'], [
        'reply_markup' => json_encode(inlineKeyboard($buttons)),
        'parse_mode' => 'Markdown'
    ]);
}

/**
 * Show submenu
 * @param TelegramBot $bot Bot instance
 * @param int $chatId Chat ID
 * @param int $messageId Message ID
 * @param int $submenuId Submenu ID
 */
function showSubmenu($bot, $chatId, $messageId, $submenuId) {
    global $db;
    
    // Get submenu
    $submenu = getSubmenu($db, $submenuId);
    if (!$submenu) {
        $bot->editMessageText($chatId, $messageId, MESSAGES['not_found']);
        return;
    }
    
    // Get content for this submenu
    $stmt = $db->prepare("SELECT * FROM content WHERE submenu_id = ? ORDER BY position");
    $stmt->execute([$submenuId]);
    $contents = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get links for this submenu
    $links = getLinksBySubmenu($db, $submenuId);
    
    // Create buttons
    $buttons = [];
    
    // Content buttons
    foreach ($contents as $content) {
        $buttons[] = [['text' => $content['title'], 'callback_data' => 'content:' . $content['id']]];
    }
    
    // Link buttons
    foreach ($links as $link) {
        if ($link['type'] == 'url') {
            $buttons[] = [['text' => $link['title'], 'url' => $link['url']]];
        } elseif ($link['type'] == 'whatsapp') {
            $buttons[] = [['text' => $link['title'], 'url' => $link['url']]];
        }
    }
    
    // Add back button
    $buttons[] = [['text' => '« Back', 'callback_data' => 'back:' . $submenu['menu_id']]];
    
    // Edit message
    $bot->editMessageText($chatId, $messageId, $submenu['title'], [
        'reply_markup' => json_encode(inlineKeyboard($buttons)),
        'parse_mode' => 'Markdown'
    ]);
}

/**
 * Show content
 * @param TelegramBot $bot Bot instance
 * @param int $chatId Chat ID
 * @param int $messageId Message ID
 * @param int $contentId Content ID
 */
function showContent($bot, $chatId, $messageId, $contentId) {
    global $db;
    
    // Get content
    $content = getContent($db, $contentId);
    if (!$content) {
        $bot->editMessageText($chatId, $messageId, MESSAGES['not_found']);
        return;
    }
    
    // Get submenu for back button
    $submenu = getSubmenu($db, $content['submenu_id']);
    
    // Create buttons
    $buttons = [
        [['text' => '« Back', 'callback_data' => 'submenu:' . $content['submenu_id']]]
    ];
    
    // Edit message
    $bot->editMessageText($chatId, $messageId, $content['content'], [
        'reply_markup' => json_encode(inlineKeyboard($buttons)),
        'parse_mode' => 'Markdown'
    ]);
}

/**
 * Process message
 * @param TelegramBot $bot Bot instance
 * @param int $chatId Chat ID
 * @param string $text Message text
 * @param array $user User data
 */
function processMessage($bot, $chatId, $text, $user) {
    global $db;
    
    // Check if in maintenance mode
    $maintenanceMode = getSetting($db, 'maintenance_mode', false);
    if ($maintenanceMode && !isAdmin($user['id'])) {
        $maintenanceMessage = getSetting($db, 'maintenance_message', 'Bot is currently under maintenance. Please try again later.');
        $bot->sendMessage($chatId, $maintenanceMessage);
        return;
    }
    
    // Show main menu by default
    showMainMenu($bot, $chatId);
    
    // Log activity
    logActivity('user', $user['id'], null, 'message', $text, $_SERVER['REMOTE_ADDR'] ?? '');
}