<?php
/**
 * Test script for Telegram Bot
 * 
 * This script allows testing the bot functionality without using the webhook
 * It can be run from the command line: php test_bot.php
 */

// Include required files
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/Database.php';
require_once __DIR__ . '/includes/Logger.php';
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/classes/TelegramBot.php';

// Initialize logger
$logger = new Logger();
$logger->info('Starting bot test script');

// Get database connection
$db = getDbConnection();

// Initialize bot
$bot = new TelegramBot($config['bot']['token'], $db, $config['bot']['debug']);

// Test functions
function testBotInfo($bot, $logger) {
    $logger->info('Testing getMe function');
    $result = $bot->getMe();
    
    if (isset($result['ok']) && $result['ok']) {
        $logger->info('Bot information retrieved successfully');
        echo "Bot Information:\n";
        echo "Username: @{$result['result']['username']}\n";
        echo "Name: {$result['result']['first_name']}\n";
        echo "ID: {$result['result']['id']}\n";
        echo "Can join groups: " . ($result['result']['can_join_groups'] ? 'Yes' : 'No') . "\n";
        echo "Can read group messages: " . ($result['result']['can_read_all_group_messages'] ? 'Yes' : 'No') . "\n";
        echo "Supports inline queries: " . ($result['result']['supports_inline_queries'] ? 'Yes' : 'No') . "\n";
        return true;
    } else {
        $logger->error('Failed to get bot information: ' . json_encode($result));
        echo "Error: Failed to get bot information\n";
        return false;
    }
}

function testWebhookInfo($bot, $logger) {
    $logger->info('Testing getWebhookInfo function');
    $result = $bot->getWebhookInfo();
    
    if (isset($result['ok']) && $result['ok']) {
        $logger->info('Webhook information retrieved successfully');
        echo "\nWebhook Information:\n";
        echo "URL: {$result['result']['url']}\n";
        echo "Has custom certificate: " . (isset($result['result']['has_custom_certificate']) && $result['result']['has_custom_certificate'] ? 'Yes' : 'No') . "\n";
        echo "Pending update count: {$result['result']['pending_update_count']}\n";
        
        if (isset($result['result']['last_error_date'])) {
            $date = date('Y-m-d H:i:s', $result['result']['last_error_date']);
            echo "Last error date: {$date}\n";
            echo "Last error message: {$result['result']['last_error_message']}\n";
        }
        
        return true;
    } else {
        $logger->error('Failed to get webhook information: ' . json_encode($result));
        echo "Error: Failed to get webhook information\n";
        return false;
    }
}

function testDatabaseConnection($db, $logger) {
    $logger->info('Testing database connection');
    
    try {
        // Try to query the settings table
        $stmt = $db->prepare("SELECT * FROM settings LIMIT 1");
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result) {
            $logger->info('Database connection successful');
            echo "\nDatabase Connection: Successful\n";
            echo "Settings found: " . count($result) . " entries\n";
            return true;
        } else {
            $logger->warning('Database connection successful but no settings found');
            echo "\nDatabase Connection: Successful but no settings found\n";
            return true;
        }
    } catch (PDOException $e) {
        $logger->error('Database connection failed: ' . $e->getMessage());
        echo "\nDatabase Connection: Failed\n";
        echo "Error: " . $e->getMessage() . "\n";
        return false;
    }
}

function testMenuStructure($db, $logger) {
    $logger->info('Testing menu structure');
    
    try {
        // Get menus
        $stmt = $db->prepare("SELECT * FROM menus");
        $stmt->execute();
        $menus = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get submenus
        $stmt = $db->prepare("SELECT * FROM submenus");
        $stmt->execute();
        $submenus = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get content
        $stmt = $db->prepare("SELECT * FROM content");
        $stmt->execute();
        $content = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo "\nMenu Structure:\n";
        echo "Menus: " . count($menus) . "\n";
        echo "Submenus: " . count($submenus) . "\n";
        echo "Content items: " . count($content) . "\n";
        
        // Display menu structure
        if (count($menus) > 0) {
            echo "\nMenu Hierarchy:\n";
            foreach ($menus as $menu) {
                echo "- {$menu['name']}\n";
                
                // Find submenus for this menu
                $menuSubmenus = array_filter($submenus, function($submenu) use ($menu) {
                    return $submenu['menu_id'] == $menu['id'];
                });
                
                foreach ($menuSubmenus as $submenu) {
                    echo "  - {$submenu['name']}\n";
                    
                    // Find content for this submenu
                    $submenuContent = array_filter($content, function($item) use ($submenu) {
                        return $item['submenu_id'] == $submenu['id'];
                    });
                    
                    foreach ($submenuContent as $item) {
                        echo "    - {$item['title']}\n";
                    }
                }
            }
        }
        
        $logger->info('Menu structure test completed');
        return true;
    } catch (PDOException $e) {
        $logger->error('Menu structure test failed: ' . $e->getMessage());
        echo "\nMenu Structure Test: Failed\n";
        echo "Error: " . $e->getMessage() . "\n";
        return false;
    }
}

function testSendMessage($bot, $logger) {
    global $config;
    
    // Only test if admin ID is set
    if (!isset($config['admin']['ids'][0]) || empty($config['admin']['ids'][0])) {
        $logger->warning('Cannot test sending message: No admin ID configured');
        echo "\nSkipping send message test: No admin ID configured\n";
        return false;
    }
    
    $adminId = $config['admin']['ids'][0];
    $logger->info("Testing sending message to admin ID: {$adminId}");
    
    echo "\nDo you want to send a test message to admin (ID: {$adminId})? (y/n): ";
    $handle = fopen("php://stdin", "r");
    $line = trim(fgets($handle));
    fclose($handle);
    
    if (strtolower($line) != 'y') {
        echo "Test message sending skipped\n";
        return false;
    }
    
    $result = $bot->sendMessage($adminId, "This is a test message from the bot test script. Time: " . date('Y-m-d H:i:s'));
    
    if (isset($result['ok']) && $result['ok']) {
        $logger->info('Test message sent successfully');
        echo "Test message sent successfully\n";
        return true;
    } else {
        $logger->error('Failed to send test message: ' . json_encode($result));
        echo "Error: Failed to send test message\n";
        return false;
    }
}

// Run tests
echo "=== Telegram Bot Test Script ===\n";

$botInfoSuccess = testBotInfo($bot, $logger);
if ($botInfoSuccess) {
    testWebhookInfo($bot, $logger);
    testDatabaseConnection($db, $logger);
    testMenuStructure($db, $logger);
    testSendMessage($bot, $logger);
}

echo "\n=== Test Completed ===\n";
$logger->info('Bot test script completed');