<?php
/**
 * Database Connection Test Script
 */

// Enable error reporting for diagnostic purposes
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include configuration
require_once __DIR__ . '/config/config.php';

// Set content type to HTML
header('Content-Type: text/html; charset=utf-8');

// Start HTML output
echo '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Database Connection Test</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            margin: 20px;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }
        h1, h2 {
            color: #333;
        }
        .success {
            color: green;
            background-color: #e8f5e9;
            padding: 10px;
            border-left: 5px solid green;
            margin-bottom: 10px;
        }
        .error {
            color: #721c24;
            background-color: #f8d7da;
            padding: 10px;
            border-left: 5px solid #f5c6cb;
            margin-bottom: 10px;
        }
        .info {
            color: #0c5460;
            background-color: #d1ecf1;
            padding: 10px;
            border-left: 5px solid #bee5eb;
            margin-bottom: 10px;
        }
        pre {
            background-color: #f5f5f5;
            padding: 10px;
            border-radius: 5px;
            overflow-x: auto;
        }
    </style>
</head>
<body>
    <h1>Database Connection Test</h1>
    
    <h2>Configuration</h2>
    <div class="info">
        <p>Host: <?php echo DB_HOST; ?></p>
        <p>Database: <?php echo DB_NAME; ?></p>
        <p>User: <?php echo DB_USER; ?></p>
        <p>Password: <?php echo !empty(DB_PASS) ? '******' : '<em>empty</em>'; ?></p>
        <p>Charset: <?php echo DB_CHARSET; ?></p>
    </div>
    
    <h2>Connection Test</h2>
    <?php
    try {
        // Create DSN
        $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
        
        // Set options
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ];
        
        // Create connection
        $db = new PDO($dsn, DB_USER, DB_PASS, $options);
        
        echo '<div class="success">Database connection successful!</div>';
        
        // Check if database exists
        $stmt = $db->query("SELECT DATABASE()");
        $currentDb = $stmt->fetchColumn();
        
        echo '<div class="info">Connected to database: ' . htmlspecialchars($currentDb) . '</div>';
        
        // Check tables
        $stmt = $db->query("SHOW TABLES");
        $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        if (count($tables) > 0) {
            echo '<div class="success">Found ' . count($tables) . ' tables in the database.</div>';
            echo '<pre>';
            print_r($tables);
            echo '</pre>';
        } else {
            echo '<div class="error">No tables found in the database. You may need to import the database schema.</div>';
        }
        
    } catch (PDOException $e) {
        echo '<div class="error">Connection failed: ' . htmlspecialchars($e->getMessage()) . '</div>';
        
        // Check if it's an access denied error
        if (strpos($e->getMessage(), 'Access denied') !== false) {
            echo '<div class="info">This appears to be a database access issue. Please check your credentials.</div>';
        }
        
        // Check if it's a database not found error
        if (strpos($e->getMessage(), "Unknown database") !== false) {
            echo '<div class="info">The database does not exist. You need to create it first.</div>';
            
            try {
                // Try to connect without database
                $dsn = "mysql:host=" . DB_HOST . ";charset=" . DB_CHARSET;
                $db = new PDO($dsn, DB_USER, DB_PASS, $options);
                
                echo '<div class="success">Connection to MySQL server successful. You can create the database.</div>';
                
                // Try to create database
                $stmt = $db->prepare("CREATE DATABASE IF NOT EXISTS `" . DB_NAME . "`");
                if ($stmt->execute()) {
                    echo '<div class="success">Database created successfully! Refresh this page to test the connection again.</div>';
                } else {
                    echo '<div class="error">Failed to create database.</div>';
                }
                
            } catch (PDOException $e2) {
                echo '<div class="error">Failed to connect to MySQL server: ' . htmlspecialchars($e2->getMessage()) . '</div>';
            }
        }
    }
    ?>
    
    <h2>Next Steps</h2>
    <p>If the database connection is successful but no tables are found, you need to import the database schema:</p>
    <ol>
        <li>Use phpMyAdmin or MySQL command line to import the schema.sql file</li>
        <li>Or create the required tables manually</li>
    </ol>
    
    <p>If the connection failed:</p>
    <ol>
        <li>Check that MySQL server is running</li>
        <li>Verify the database credentials in config.php</li>
        <li>Make sure the database exists</li>
        <li>Ensure the user has proper permissions</li>
    </ol>
    
    <p><a href="/new_bot/admin/">Go to Admin Panel</a></p>
</body>
</html>';