<?php
/**
 * Test connection to telegram_bot_admin database
 * 
 * This script specifically tests connection to the telegram_bot_admin database
 * that was mentioned in the error message.
 */

// Enable error reporting for diagnostic purposes
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Load configuration
require_once __DIR__ . '/config/config.php';

echo "<h1>Testing Connection to telegram_bot_admin Database</h1>";

// Function to test database connection
function testDatabaseConnection($host, $user, $pass, $name = null) {
    try {
        $dsn = "mysql:host={$host}" . ($name ? ";dbname={$name}" : '');
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ];
        
        $db = new PDO($dsn, $user, $pass, $options);
        return ['success' => true, 'message' => 'Connection successful'];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Function to check if user has database creation privileges
function checkUserPrivileges($host, $user, $pass) {
    try {
        $dsn = "mysql:host={$host}";
        $db = new PDO($dsn, $user, $pass);
        
        $stmt = $db->query("SHOW GRANTS FOR CURRENT_USER()");
        $grants = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        $hasCreateDb = false;
        $hasAllPrivileges = false;
        
        foreach ($grants as $grant) {
            if (strpos($grant, 'ALL PRIVILEGES ON *.*') !== false) {
                $hasAllPrivileges = true;
            }
            if (strpos($grant, 'CREATE') !== false) {
                $hasCreateDb = true;
            }
        }
        
        return [
            'success' => true,
            'has_create_db' => $hasCreateDb || $hasAllPrivileges,
            'has_all_privileges' => $hasAllPrivileges,
            'grants' => $grants
        ];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Function to check if database exists
function checkDatabaseExists($host, $user, $pass, $name) {
    try {
        $dsn = "mysql:host={$host}";
        $db = new PDO($dsn, $user, $pass);
        
        $stmt = $db->query("SELECT SCHEMA_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = '{$name}'");
        $result = $stmt->fetchColumn();
        
        return ['success' => true, 'exists' => !empty($result)];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Function to create database
function createDatabase($host, $user, $pass, $name) {
    try {
        $dsn = "mysql:host={$host}";
        $db = new PDO($dsn, $user, $pass);
        
        // Create database if it doesn't exist
        $db->exec("CREATE DATABASE IF NOT EXISTS `{$name}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        
        return ['success' => true, 'message' => "Database '{$name}' created successfully"];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Display configuration
echo "<h2>Current Database Configuration</h2>";
echo "<ul>";
echo "<li><strong>Host:</strong> " . DB_HOST . "</li>";
echo "<li><strong>Database:</strong> " . DB_NAME . "</li>";
echo "<li><strong>User:</strong> " . DB_USER . "</li>";
echo "<li><strong>Password:</strong> " . str_repeat('*', strlen(DB_PASS)) . "</li>";
echo "</ul>";

// Test connection to MySQL server (without database)
echo "<h2>Testing MySQL Server Connection</h2>";
$serverConnection = testDatabaseConnection(DB_HOST, DB_USER, DB_PASS);

if ($serverConnection['success']) {
    echo "<p style='color:green'>✅ Successfully connected to MySQL server</p>";
    
    // Check user privileges
    $privileges = checkUserPrivileges(DB_HOST, DB_USER, DB_PASS);
    
    if ($privileges['success']) {
        echo "<h2>User Privileges</h2>";
        
        if ($privileges['has_all_privileges']) {
            echo "<p style='color:green'>✅ User has ALL PRIVILEGES</p>";
        } elseif ($privileges['has_create_db']) {
            echo "<p style='color:green'>✅ User has CREATE DATABASE privilege</p>";
        } else {
            echo "<p style='color:orange'>⚠️ User does not have CREATE DATABASE privilege</p>";
        }
        
        echo "<h3>Grants:</h3>";
        echo "<ul>";
        foreach ($privileges['grants'] as $grant) {
            echo "<li>{$grant}</li>";
        }
        echo "</ul>";
        
        // Check if telegram_bot_admin database exists
        $dbName = 'telegram_bot_admin';
        $dbExists = checkDatabaseExists(DB_HOST, DB_USER, DB_PASS, $dbName);
        
        if ($dbExists['success']) {
            echo "<h2>Checking '{$dbName}' Database</h2>";
            
            if ($dbExists['exists']) {
                echo "<p style='color:green'>✅ Database '{$dbName}' exists</p>";
                
                // Test connection to the database
                $dbConnection = testDatabaseConnection(DB_HOST, DB_USER, DB_PASS, $dbName);
                
                if ($dbConnection['success']) {
                    echo "<p style='color:green'>✅ Successfully connected to database '{$dbName}'</p>";
                } else {
                    echo "<p style='color:red'>❌ Failed to connect to database '{$dbName}': {$dbConnection['message']}</p>";
                    
                    // Check if it's an access denied error
                    if (strpos($dbConnection['message'], 'Access denied') !== false) {
                        echo "<p>This is an access denied error. The database exists but the user does not have access to it.</p>";
                        echo "<p><strong>Solution:</strong> An administrator needs to grant privileges to user '" . DB_USER . "' on database '{$dbName}'</p>";
                    }
                }
            } else {
                echo "<p style='color:orange'>⚠️ Database '{$dbName}' does not exist</p>";
                
                if ($privileges['has_create_db']) {
                    echo "<p>User has permission to create the database</p>";
                    
                    // Attempt to create the database
                    echo "<h3>Attempting to Create Database</h3>";
                    $createResult = createDatabase(DB_HOST, DB_USER, DB_PASS, $dbName);
                    
                    if ($createResult['success']) {
                        echo "<p style='color:green'>✅ {$createResult['message']}</p>";
                    } else {
                        echo "<p style='color:red'>❌ Failed to create database: {$createResult['message']}</p>";
                    }
                } else {
                    echo "<p style='color:red'>❌ User does not have permission to create the database</p>";
                    echo "<p><strong>Solution:</strong> You need MySQL administrator privileges to create this database.</p>";
                    echo "<p>Run the following SQL commands as a MySQL administrator:</p>";
                    echo "<pre>CREATE DATABASE IF NOT EXISTS `{$dbName}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;\nGRANT ALL PRIVILEGES ON `{$dbName}`.* TO '" . DB_USER . "'@'localhost';\nFLUSH PRIVILEGES;</pre>";
                }
            }
        } else {
            echo "<p style='color:red'>❌ Failed to check if database exists: {$dbExists['message']}</p>";
        }
    } else {
        echo "<p style='color:red'>❌ Failed to check user privileges: {$privileges['message']}</p>";
    }
} else {
    echo "<p style='color:red'>❌ Failed to connect to MySQL server: {$serverConnection['message']}</p>";
}

// Compare with configured database
echo "<h2>Comparison with Configured Database</h2>";
echo "<p>Your application is configured to use database <strong>" . DB_NAME . "</strong>, but you're trying to create <strong>telegram_bot_admin</strong>.</p>";

if (DB_NAME !== 'telegram_bot_admin') {
    echo "<p style='color:orange'>⚠️ These are different databases. Make sure you're using the correct database name in your SQL commands.</p>";
    echo "<p><strong>Options:</strong></p>";
    echo "<ol>";
    echo "<li>Change your SQL command to create <strong>" . DB_NAME . "</strong> instead of telegram_bot_admin</li>";
    echo "<li>Update your config.php to use telegram_bot_admin as the database name</li>";
    echo "</ol>";
} else {
    echo "<p style='color:green'>✅ Your configuration matches the database you're trying to create.</p>";
}