<?php
/**
 * Update Database Configuration
 * 
 * This script updates the config.php file to use the telegram_bot_admin database
 * or another database name of your choice.
 */

// Enable error reporting for diagnostic purposes
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set content type to HTML
header('Content-Type: text/html; charset=utf-8');

// Start HTML output
echo '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Update Database Configuration</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            margin: 20px;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }
        h1, h2, h3 {
            color: #333;
        }
        .success {
            color: green;
            background-color: #e8f5e9;
            padding: 10px;
            border-left: 5px solid green;
            margin-bottom: 10px;
        }
        .warning {
            color: #856404;
            background-color: #fff3cd;
            padding: 10px;
            border-left: 5px solid #ffeeba;
            margin-bottom: 10px;
        }
        .error {
            color: #721c24;
            background-color: #f8d7da;
            padding: 10px;
            border-left: 5px solid #f5c6cb;
            margin-bottom: 10px;
        }
        .info {
            color: #0c5460;
            background-color: #d1ecf1;
            padding: 10px;
            border-left: 5px solid #bee5eb;
            margin-bottom: 10px;
        }
        pre {
            background-color: #f5f5f5;
            padding: 10px;
            border-radius: 5px;
            overflow-x: auto;
        }
        code {
            font-family: monospace;
            background-color: #f5f5f5;
            padding: 2px 5px;
            border-radius: 3px;
        }
        .section {
            margin-bottom: 30px;
            border: 1px solid #ddd;
            padding: 20px;
            border-radius: 5px;
        }
        form {
            margin-bottom: 20px;
        }
        input[type=text], input[type=password] {
            width: 100%;
            padding: 12px 20px;
            margin: 8px 0;
            display: inline-block;
            border: 1px solid #ccc;
            border-radius: 4px;
            box-sizing: border-box;
        }
        .form-group {
            margin-bottom: 15px;
        }
        label {
            font-weight: bold;
        }
        button {
            background-color: #4CAF50;
            color: white;
            padding: 10px 15px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 16px;
        }
        button:hover {
            background-color: #45a049;
        }
        .diff {
            background-color: #f8f9fa;
            border: 1px solid #ddd;
            padding: 10px;
            margin: 10px 0;
            font-family: monospace;
            white-space: pre-wrap;
        }
        .diff-added {
            background-color: #e6ffed;
            color: #22863a;
        }
        .diff-removed {
            background-color: #ffeef0;
            color: #cb2431;
        }
    </style>
</head>
<body>
    <h1>Update Database Configuration</h1>
    <p>This tool helps you update your database configuration in config.php to use the telegram_bot_admin database or another database name of your choice.</p>
';

// Function to get current config settings
function getCurrentConfig() {
    $configFile = dirname(__DIR__) . '/config/config.php';
    
    if (!file_exists($configFile)) {
        echo "<div class='error'>❌ Configuration file not found: {$configFile}</div>";
        return false;
    }
    
    // Read the file content
    $configContent = file_get_contents($configFile);
    
    // Extract database settings using regex
    $host = '';
    $name = '';
    $user = '';
    $pass = '';
    
    if (preg_match("/define\('DB_HOST', '(.*?)'\);/", $configContent, $matches)) {
        $host = $matches[1];
    }
    
    if (preg_match("/define\('DB_NAME', '(.*?)'\);/", $configContent, $matches)) {
        $name = $matches[1];
    }
    
    if (preg_match("/define\('DB_USER', '(.*?)'\);/", $configContent, $matches)) {
        $user = $matches[1];
    }
    
    if (preg_match("/define\('DB_PASS', '(.*?)'\);/", $configContent, $matches)) {
        $pass = $matches[1];
    }
    
    return [
        'host' => $host,
        'name' => $name,
        'user' => $user,
        'pass' => $pass,
        'content' => $configContent
    ];
}

// Function to update config file
function updateConfig($configContent, $newDbName) {
    $configFile = dirname(__DIR__) . '/config/config.php';
    
    // Replace database name
    $updatedContent = preg_replace("/define\('DB_NAME', '(.*?)'\);/", "define('DB_NAME', '{$newDbName}');", $configContent);
    
    // Create backup
    $backupFile = $configFile . '.bak.' . date('YmdHis');
    if (!copy($configFile, $backupFile)) {
        return ['success' => false, 'message' => 'Failed to create backup file'];
    }
    
    // Write updated content
    if (file_put_contents($configFile, $updatedContent)) {
        return ['success' => true, 'message' => 'Configuration updated successfully', 'backup' => $backupFile];
    } else {
        return ['success' => false, 'message' => 'Failed to write to configuration file'];
    }
}

// Function to test database connection
function testDatabaseConnection($host, $user, $pass, $name = null) {
    try {
        $dsn = "mysql:host={$host}" . ($name ? ";dbname={$name}" : '');
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ];
        
        $db = new PDO($dsn, $user, $pass, $options);
        return ['success' => true, 'message' => 'Connection successful'];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Main execution
echo "<div class='section'>";

// Get current configuration
$config = getCurrentConfig();

if ($config) {
    echo "<h2>Current Configuration</h2>";
    echo "<ul>";
    echo "<li><strong>Host:</strong> {$config['host']}</li>";
    echo "<li><strong>Database:</strong> {$config['name']}</li>";
    echo "<li><strong>User:</strong> {$config['user']}</li>";
    echo "<li><strong>Password:</strong> " . str_repeat('*', strlen($config['pass'])) . "</li>";
    echo "</ul>";
    
    // Display update form
    echo "<h2>Update Database Name</h2>";
    echo "<form method='post' action=''>";
    echo "<div class='form-group'>";
    echo "<label for='new_db_name'>New Database Name:</label>";
    echo "<input type='text' id='new_db_name' name='new_db_name' value='telegram_bot_admin' required>";
    echo "</div>";
    echo "<button type='submit' name='update'>Update Configuration</button>";
    echo "</form>";
}

echo "</div>";

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update'])) {
    $newDbName = $_POST['new_db_name'];
    
    echo "<div class='section'><h2>Update Results</h2>";
    
    // Test connection to the new database
    echo "<h3>Testing Connection to '{$newDbName}'</h3>";
    $testResult = testDatabaseConnection($config['host'], $config['user'], $config['pass'], $newDbName);
    
    if ($testResult['success']) {
        echo "<div class='success'>✅ Connection to database '{$newDbName}' successful</div>";
        echo "<p>The database exists and your user has access to it.</p>";
    } else {
        echo "<div class='warning'>⚠️ Connection to database '{$newDbName}' failed: {$testResult['message']}</div>";
        
        if (strpos($testResult['message'], "Unknown database") !== false) {
            echo "<p>The database does not exist. You need to create it first.</p>";
        } elseif (strpos($testResult['message'], "Access denied") !== false) {
            echo "<p>The database exists but your user does not have access to it.</p>";
        }
        
        echo "<p>Do you still want to update the configuration?</p>";
        echo "<form method='post' action=''>";
        echo "<input type='hidden' name='new_db_name' value='{$newDbName}'>";
        echo "<button type='submit' name='force_update'>Yes, Update Anyway</button>";
        echo "</form>";
        
        echo "</div>";
        echo "</body></html>";
        exit;
    }
    
    // Update configuration
    $updateResult = updateConfig($config['content'], $newDbName);
    
    if ($updateResult['success']) {
        echo "<div class='success'>✅ {$updateResult['message']}</div>";
        echo "<p>A backup of your original configuration has been created at: {$updateResult['backup']}</p>";
        
        // Show diff
        echo "<h3>Configuration Changes</h3>";
        echo "<div class='diff'>";
        echo "<div class='diff-removed'>- define('DB_NAME', '{$config['name']}');</div>";
        echo "<div class='diff-added'>+ define('DB_NAME', '{$newDbName}');</div>";
        echo "</div>";
        
        echo "<p>Your configuration has been updated to use the '{$newDbName}' database.</p>";
    } else {
        echo "<div class='error'>❌ {$updateResult['message']}</div>";
    }
    
    echo "</div>";
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['force_update'])) {
    $newDbName = $_POST['new_db_name'];
    
    echo "<div class='section'><h2>Update Results</h2>";
    
    // Update configuration
    $updateResult = updateConfig($config['content'], $newDbName);
    
    if ($updateResult['success']) {
        echo "<div class='success'>✅ {$updateResult['message']}</div>";
        echo "<p>A backup of your original configuration has been created at: {$updateResult['backup']}</p>";
        
        // Show diff
        echo "<h3>Configuration Changes</h3>";
        echo "<div class='diff'>";
        echo "<div class='diff-removed'>- define('DB_NAME', '{$config['name']}');</div>";
        echo "<div class='diff-added'>+ define('DB_NAME', '{$newDbName}');</div>";
        echo "</div>";
        
        echo "<div class='warning'>⚠️ Remember that you need to create the '{$newDbName}' database or ensure your user has access to it.</div>";
    } else {
        echo "<div class='error'>❌ {$updateResult['message']}</div>";
    }
    
    echo "</div>";
}

// Database creation instructions
echo "<div class='section'><h2>Database Creation Instructions</h2>";

echo "<h3>Option 1: Using phpMyAdmin</h3>";
echo "<ol>";
echo "<li>Log in to phpMyAdmin with your hosting control panel</li>";
echo "<li>Click on 'Databases' tab</li>";
echo "<li>Enter 'telegram_bot_admin' in the 'Create database' field</li>";
echo "<li>Select 'utf8mb4_unicode_ci' from the collation dropdown</li>";
echo "<li>Click 'Create' button</li>";
echo "<li>Go to 'Users' tab</li>";
echo "<li>Click 'Add user' or edit existing user '{$config['user']}'</li>";
echo "<li>Grant all privileges on the database 'telegram_bot_admin' to user '{$config['user']}'</li>";
echo "</ol>";

echo "<h3>Option 2: Using MySQL Command Line</h3>";
echo "<pre>";
echo "# Login as MySQL admin\nmysql -u root -p\n\n# Create database\nCREATE DATABASE IF NOT EXISTS `telegram_bot_admin` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;\n\n# Grant privileges\nGRANT ALL PRIVILEGES ON `telegram_bot_admin`.* TO '{$config['user']}'@'localhost';\n\n# Apply changes\nFLUSH PRIVILEGES;\n";
echo "</pre>";

echo "<h3>Option 3: Contact Your Hosting Provider</h3>";
echo "<p>If you don't have access to phpMyAdmin or MySQL command line, contact your hosting provider with the following information:</p>";
echo "<ul>";
echo "<li>Request to create a new database named 'telegram_bot_admin'</li>";
echo "<li>Request to grant all privileges on this database to user '{$config['user']}'</li>";
echo "<li>Ensure the database uses UTF-8 (utf8mb4) character encoding</li>";
echo "</ul>";

echo "</div>";

// End HTML output
echo '</body>
</html>';