<?php
/**
 * Configuration Verification Script
 * 
 * This script verifies the bot configuration and database connection.
 * Run this script to check if your configuration is correct.
 */

// Include configuration
require_once __DIR__ . '/config/config.php';

echo "=== Telegram Bot Configuration Verification ===\n\n";

// Check bot configuration
echo "Bot Configuration:\n";
echo "- Bot Token: " . (BOT_TOKEN ? "✓ Configured" : "✗ Missing") . "\n";
echo "- Bot Username: " . (BOT_USERNAME ? "✓ Configured (@" . BOT_USERNAME . ")" : "✗ Missing") . "\n";
echo "- Webhook URL: " . (WEBHOOK_URL ? "✓ Configured (" . WEBHOOK_URL . ")" : "✗ Missing") . "\n";

// Check admin configuration
echo "\nAdmin Configuration:\n";
echo "- Admin IDs: " . (count(ADMIN_IDS) > 0 ? "✓ Configured (" . implode(", ", ADMIN_IDS) . ")" : "✗ Missing") . "\n";
echo "- Admin Password: " . (ADMIN_PASSWORD ? "✓ Configured" : "✗ Missing") . "\n";

// Check database configuration
echo "\nDatabase Configuration:\n";
echo "- Host: " . (DB_HOST ? "✓ Configured (" . DB_HOST . ")" : "✗ Missing") . "\n";
echo "- Database Name: " . (DB_NAME ? "✓ Configured (" . DB_NAME . ")" : "✗ Missing") . "\n";
echo "- Username: " . (DB_USER ? "✓ Configured (" . DB_USER . ")" : "✗ Missing") . "\n";
echo "- Password: " . (DB_PASS ? "✓ Configured" : "✗ Missing") . "\n";

// Test database connection
echo "\nTesting Database Connection...\n";
try {
    $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    $pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
    echo "✓ Database connection successful!\n";
    
    // Check if tables exist
    $tables = ['users', 'admin_users', 'menus', 'submenus', 'content', 'links', 'broadcasts', 'sessions', 'settings', 'activity_logs'];
    $missingTables = [];
    
    foreach ($tables as $table) {
        $stmt = $pdo->prepare("SHOW TABLES LIKE :table");
        $stmt->execute(['table' => $table]);
        if ($stmt->rowCount() == 0) {
            $missingTables[] = $table;
        }
    }
    
    if (count($missingTables) > 0) {
        echo "✗ Some tables are missing: " . implode(", ", $missingTables) . "\n";
        echo "  Run the database initialization script: php database/init_db.php\n";
    } else {
        echo "✓ All required tables exist!\n";
    }
    
} catch (PDOException $e) {
    echo "✗ Database connection failed: " . $e->getMessage() . "\n";
    echo "  Please check your database configuration.\n";
}

// Test webhook URL accessibility
echo "\nTesting Webhook URL Accessibility...\n";
if (function_exists('curl_version')) {
    $ch = curl_init(WEBHOOK_URL);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_NOBODY, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode >= 200 && $httpCode < 300) {
        echo "✓ Webhook URL is accessible (HTTP {$httpCode})\n";
    } else {
        echo "✗ Webhook URL is not accessible (HTTP {$httpCode})\n";
        echo "  Please check your server configuration and ensure the URL is publicly accessible.\n";
    }
} else {
    echo "? Could not test webhook URL accessibility (curl extension not available)\n";
}

// Test Telegram Bot API connection
echo "\nTesting Telegram Bot API Connection...\n";
if (function_exists('curl_version')) {
    $ch = curl_init("https://api.telegram.org/bot" . BOT_TOKEN . "/getMe");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    $response = curl_exec($ch);
    curl_close($ch);
    
    $result = json_decode($response, true);
    
    if (isset($result['ok']) && $result['ok']) {
        echo "✓ Telegram Bot API connection successful!\n";
        echo "  Bot Name: {$result['result']['first_name']}\n";
        echo "  Bot Username: @{$result['result']['username']}\n";
        
        // Check if the configured username matches the actual username
        if ($result['result']['username'] !== BOT_USERNAME) {
            echo "✗ Warning: Configured username (@" . BOT_USERNAME . ") does not match actual bot username (@{$result['result']['username']})\n";
        }
    } else {
        echo "✗ Telegram Bot API connection failed!\n";
        echo "  Error: " . (isset($result['description']) ? $result['description'] : "Unknown error") . "\n";
        echo "  Please check your bot token.\n";
    }
} else {
    echo "? Could not test Telegram Bot API connection (curl extension not available)\n";
}

echo "\n=== Verification Complete ===\n";