<?php
/**
 * Webhook handler for Telegram Bot
 * This file receives and processes updates from Telegram
 */

// Include required files
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/database.php';
require_once __DIR__ . '/classes/TelegramBot.php';

// Create log directory if it doesn't exist
if (!file_exists(__DIR__ . '/logs')) {
    mkdir(__DIR__ . '/logs', 0755, true);
}

// Initialize bot
$bot = new TelegramBot(BOT_TOKEN, $db);

// Get the incoming update from Telegram
$update = json_decode(file_get_contents('php://input'), true);

// Log the update if debug mode is enabled
if (defined('DEBUG_MODE') && DEBUG_MODE) {
    file_put_contents(__DIR__ . '/logs/webhook.log', date('[Y-m-d H:i:s]') . ' Update: ' . json_encode($update) . PHP_EOL, FILE_APPEND);
}

// Process the update
if (!$update) {
    // No update received
    http_response_code(400);
    exit('No update received');
}

// Extract update data
if (isset($update['message'])) {
    // Regular message
    $message = $update['message'];
    $chatId = $message['chat']['id'];
    $text = $message['text'] ?? '';
    $user = $message['from'];
    
    // Save user to database
    $bot->saveUser($user);
    
    // Check if user is blocked
    $stmt = $db->prepare("SELECT is_blocked FROM users WHERE telegram_id = ?");
    $stmt->execute([$user['id']]);
    $userData = $stmt->fetch();
    
    if ($userData && $userData['is_blocked']) {
        // User is blocked
        $bot->sendMessage($chatId, MESSAGES['blocked']);
        exit;
    }
    
    // Process commands
    if (strpos($text, '/') === 0) {
        processCommand($bot, $chatId, $text, $user);
    } else {
        // Process regular message
        processMessage($bot, $chatId, $text, $user);
    }
    
} elseif (isset($update['callback_query'])) {
    // Callback query from inline keyboard
    $callbackQuery = $update['callback_query'];
    $chatId = $callbackQuery['message']['chat']['id'];
    $messageId = $callbackQuery['message']['message_id'];
    $data = $callbackQuery['data'];
    $user = $callbackQuery['from'];
    
    // Save user to database
    $bot->saveUser($user);
    
    // Process callback query
    processCallbackQuery($bot, $chatId, $messageId, $data, $callbackQuery['id'], $user);
}

// Send 200 OK response to Telegram
http_response_code(200);
exit('OK');

/**
 * Process commands
 * @param TelegramBot $bot Bot instance
 * @param int $chatId Chat ID
 * @param string $text Command text
 * @param array $user User data
 */
function processCommand($bot, $chatId, $text, $user) {
    global $db;
    
    // Extract command and parameters
    $parts = explode(' ', $text);
    $command = strtolower(array_shift($parts));
    $params = implode(' ', $parts);
    
    switch ($command) {
        case '/start':
            // Send welcome message
            $welcomeMessage = MESSAGES['welcome'];
            
            // Create main menu
            $menus = $bot->getMenus();
            $buttons = [];
            
            if (!empty($menus)) {
                foreach ($menus as $menu) {
                    $buttons[] = [[
                        'text' => $menu['title'],
                        'callback_data' => 'menu_' . $menu['id']
                    ]];
                }
            } else {
                $buttons[] = [['text' => 'No menus available', 'callback_data' => 'no_action']];
            }
            
            $keyboard = $bot->createInlineKeyboard($buttons);
            
            $bot->sendMessage($chatId, $welcomeMessage, [
                'reply_markup' => json_encode($keyboard),
                'parse_mode' => 'Markdown'
            ]);
            break;
            
        case '/help':
            // Send help message
            $helpMessage = "*Available Commands:*\n"
                . "/start - Start the bot and show main menu\n"
                . "/help - Show this help message\n"
                . "/menu - Show main menu\n";
                
            $bot->sendMessage($chatId, $helpMessage, ['parse_mode' => 'Markdown']);
            break;
            
        case '/menu':
            // Show main menu
            $menus = $bot->getMenus();
            $buttons = [];
            
            if (!empty($menus)) {
                foreach ($menus as $menu) {
                    $buttons[] = [[
                        'text' => $menu['title'],
                        'callback_data' => 'menu_' . $menu['id']
                    ]];
                }
            } else {
                $buttons[] = [['text' => 'No menus available', 'callback_data' => 'no_action']];
            }
            
            $keyboard = $bot->createInlineKeyboard($buttons);
            
            $bot->sendMessage($chatId, MESSAGES['menu_prompt'], [
                'reply_markup' => json_encode($keyboard),
                'parse_mode' => 'Markdown'
            ]);
            break;
            
        default:
            // Unknown command
            $bot->sendMessage($chatId, "Unknown command. Type /help for available commands.");
            break;
    }
}

/**
 * Process regular messages
 * @param TelegramBot $bot Bot instance
 * @param int $chatId Chat ID
 * @param string $text Message text
 * @param array $user User data
 */
function processMessage($bot, $chatId, $text, $user) {
    global $db;
    
    // Check user session
    $stmt = $db->prepare("SELECT * FROM user_sessions WHERE user_id = ?");
    $stmt->execute([$user['id']]);
    $session = $stmt->fetch();
    
    if (!$session) {
        // No active session, show main menu
        $bot->sendMessage($chatId, "Please use the menu to navigate. Type /menu to see options.");
        return;
    }
    
    // Process based on session state
    $sessionData = json_decode($session['session_data'], true) ?? [];
    
    // Default response if no specific handling
    $bot->sendMessage($chatId, "I received your message. Please use the menu to navigate.");
}

/**
 * Process callback queries
 * @param TelegramBot $bot Bot instance
 * @param int $chatId Chat ID
 * @param int $messageId Message ID
 * @param string $data Callback data
 * @param string $queryId Callback query ID
 * @param array $user User data
 */
function processCallbackQuery($bot, $chatId, $messageId, $data, $queryId, $user) {
    global $db;
    
    // Answer callback query to remove loading indicator
    $bot->answerCallbackQuery($queryId);
    
    // Process based on callback data
    if (strpos($data, 'menu_') === 0) {
        // Menu selection
        $menuId = substr($data, 5);
        
        // Get menu details
        $stmt = $db->prepare("SELECT * FROM menus WHERE id = ?");
        $stmt->execute([$menuId]);
        $menu = $stmt->fetch();
        
        if (!$menu) {
            $bot->sendMessage($chatId, MESSAGES['not_found']);
            return;
        }
        
        // Get submenus
        $submenus = $bot->getSubmenus($menuId);
        $buttons = [];
        
        if (!empty($submenus)) {
            foreach ($submenus as $submenu) {
                $buttons[] = [[
                    'text' => $submenu['title'],
                    'callback_data' => 'submenu_' . $submenu['id']
                ]];
            }
        } else {
            $buttons[] = [['text' => 'No submenus available', 'callback_data' => 'no_action']];
        }
        
        // Add back button
        $buttons[] = [['text' => '« Back to Main Menu', 'callback_data' => 'back_to_main']];
        
        $keyboard = $bot->createInlineKeyboard($buttons);
        
        // Update message with submenu
        $bot->editMessageText($chatId, $messageId, "*{$menu['title']}*\n{$menu['description']}", [
            'reply_markup' => json_encode($keyboard),
            'parse_mode' => 'Markdown'
        ]);
        
        // Update user session
        updateUserSession($user['id'], ['current_menu_id' => $menuId]);
        
    } elseif (strpos($data, 'submenu_') === 0) {
        // Submenu selection
        $submenuId = substr($data, 8);
        
        // Get submenu details
        $stmt = $db->prepare("SELECT s.*, m.title as menu_title FROM submenus s 
                            JOIN menus m ON s.menu_id = m.id 
                            WHERE s.id = ?");
        $stmt->execute([$submenuId]);
        $submenu = $stmt->fetch();
        
        if (!$submenu) {
            $bot->sendMessage($chatId, MESSAGES['not_found']);
            return;
        }
        
        // Get content
        $content = $bot->getContent($submenuId);
        $links = $bot->getLinks($submenuId);
        
        $messageText = "*{$submenu['title']}*\n{$submenu['description']}";
        
        // Add content
        if (!empty($content)) {
            foreach ($content as $item) {
                $messageText .= "\n\n*{$item['title']}*\n{$item['content_text']}";
            }
        }
        
        // Create buttons for links
        $buttons = [];
        
        if (!empty($links)) {
            foreach ($links as $link) {
                $buttons[] = [[
                    'text' => $link['title'],
                    'url' => $link['url']
                ]];
            }
        }
        
        // Add back button
        $buttons[] = [['text' => '« Back to ' . $submenu['menu_title'], 'callback_data' => 'menu_' . $submenu['menu_id']]];
        
        $keyboard = $bot->createInlineKeyboard($buttons);
        
        // Update message with content
        $bot->editMessageText($chatId, $messageId, $messageText, [
            'reply_markup' => json_encode($keyboard),
            'parse_mode' => 'Markdown',
            'disable_web_page_preview' => true
        ]);
        
        // Update user session
        updateUserSession($user['id'], [
            'current_menu_id' => $submenu['menu_id'],
            'current_submenu_id' => $submenuId
        ]);
        
    } elseif ($data === 'back_to_main') {
        // Back to main menu
        $menus = $bot->getMenus();
        $buttons = [];
        
        if (!empty($menus)) {
            foreach ($menus as $menu) {
                $buttons[] = [[
                    'text' => $menu['title'],
                    'callback_data' => 'menu_' . $menu['id']
                ]];
            }
        } else {
            $buttons[] = [['text' => 'No menus available', 'callback_data' => 'no_action']];
        }
        
        $keyboard = $bot->createInlineKeyboard($buttons);
        
        $bot->editMessageText($chatId, $messageId, MESSAGES['menu_prompt'], [
            'reply_markup' => json_encode($keyboard),
            'parse_mode' => 'Markdown'
        ]);
        
        // Update user session
        updateUserSession($user['id'], ['current_menu_id' => null, 'current_submenu_id' => null]);
    }
}

/**
 * Update user session
 * @param int $userId User ID
 * @param array $data Session data to update
 */
function updateUserSession($userId, $data) {
    global $db;
    
    try {
        // Check if session exists
        $stmt = $db->prepare("SELECT * FROM user_sessions WHERE user_id = ?");
        $stmt->execute([$userId]);
        $session = $stmt->fetch();
        
        if ($session) {
            // Update existing session
            $sessionData = json_decode($session['session_data'], true) ?? [];
            $sessionData = array_merge($sessionData, $data);
            
            $stmt = $db->prepare("UPDATE user_sessions SET 
                session_data = ?, 
                current_menu_id = ?, 
                current_submenu_id = ?, 
                last_activity = NOW() 
                WHERE user_id = ?");
            $stmt->execute([
                json_encode($sessionData),
                $data['current_menu_id'] ?? $session['current_menu_id'],
                $data['current_submenu_id'] ?? $session['current_submenu_id'],
                $userId
            ]);
        } else {
            // Create new session
            $stmt = $db->prepare("INSERT INTO user_sessions 
                (user_id, session_data, current_menu_id, current_submenu_id) 
                VALUES (?, ?, ?, ?)");
            $stmt->execute([
                $userId,
                json_encode($data),
                $data['current_menu_id'] ?? null,
                $data['current_submenu_id'] ?? null
            ]);
        }
        
        return true;
    } catch (Exception $e) {
        logError("Failed to update user session: " . $e->getMessage());
        return false;
    }
}